#include "PShader.hpp"

#include <GL/glew.h>

#include <iostream>

#ifdef _DEBUG
	#include <cassert>
#endif

PShader :: PShader(void)
{
	psId = glCreateProgram();
#ifdef _DEBUG
	if ( psId == 0 )
		std::cerr << "Erreur while creating the identifier of the program (possibly executed between glBegin() and glEnd())" << std::endl;
	state = NotAttached;
#endif
}

PShader :: ~PShader(void)
{
	glDeleteProgram(psId);
}

void PShader :: attach(unsigned int shaderId)
{
	glAttachShader(psId, shaderId);

#ifdef _DEBUG
	state = NotLinked;
#endif
}

void PShader :: detach(unsigned int shaderId)
{
#ifdef _DEBUG
	assert(state!=NotAttached);
#endif
	
	glDetachShader(psId, shaderId);
#ifdef _DEBUG
	state = NotAttached;
#endif
}

bool PShader :: link(void)
{
#ifdef _DEBUG
	assert(state == NotLinked);
#endif
	
	GLint linkStatus = GL_TRUE;
	
	glLinkProgram(psId);
	
	glGetProgramiv(psId, GL_LINK_STATUS, &linkStatus);
	if ( linkStatus != GL_TRUE)
	{
		std::cerr << "Error during the link of the shader" << std::endl;
		
		GLint errorStringSize = 0;
		char* errorString = NULL;
		
		glGetProgramiv(psId, GL_INFO_LOG_LENGTH, &errorStringSize);
		
		errorString = new char[errorStringSize+1];
		if ( errorString == NULL )
		{
			std::cerr << "Error while allocating memory to get the log entry of OpenGL" << std::endl;
			return false;
		}
		
		glGetProgramInfoLog(psId, errorStringSize, &errorStringSize, errorString);
		std::cerr << "=> " << errorString << std::endl;
		
		delete []errorString;
		
		return false;
	}
#ifdef _DEBUG
	state = Ready;
#endif
	return true;
}

void PShader :: use(void)
{
#ifdef _DEBUG
	assert (state == Ready);
#endif
	
	glUseProgram(psId);
#ifdef _DEBUG
	state = InUse;
#endif
}

void PShader :: unuse(void)
{
#ifdef _DEBUG
	assert(state == InUse);
#endif
	
	glUseProgram(0);
#ifdef _DEBUG
	state = Ready;
#endif
}

int PShader :: getUniformId(const std::string& name)const
{
	int id = glGetUniformLocation(psId, name.c_str());
#ifdef _DEBUG
	if ( id == -1 )
		std::cerr << "The uniform id can not be reached [Variable passed: " << name.c_str() << std::endl;
#endif
	return id;
}

int PShader :: getAttribId(const std::string& name)const
{
	int id = glGetAttribLocation(psId, name.c_str());
#ifdef _DEBUG
	if ( id == -1 )
		std::cerr << "The attribute id can not be reached [Variable passed: " << name.c_str() << std::endl;
#endif
	return id;
}
