#include "Scene.hpp"

#include <GL/glew.h>
#include <GL/freeglut.h>
#include <GL/gl.h>

#include "Camera/Camera.hpp"

#include "Shaders/VShader.hpp"
#include "Shaders/FShader.hpp"
#include "Shaders/PShader.hpp"

#include "Particles/ParticleSystem.hpp"

#include "Sound/Music.hpp"

#include "Texts/Text.hpp"

#include "Types/Colours.hpp"

#include "Utils/String.hpp"

#include "Callbacks/Close.hpp"

#include <iostream>

Scene :: Scene (void)
{	
	cam = new Camera();
	
	musicSpectrum = new float[SPECTRUM_PRECISION];
	memset(musicSpectrum, 0, sizeof(float)*SPECTRUM_PRECISION);
	
	vertexShader = new VShader();
	fSBlinking = new FShader();
	programShader = new PShader();

	readyToQuit = false;
	
	std::cout << "Scene created" << std::endl;
}

Scene :: ~Scene(void)
{
	delete cam;
	
	delete vertexShader;
	delete fSBlinking;
	delete programShader;
	
	delete []musicSpectrum;
	
	for ( std::vector<ParticleSystem*>::iterator it_particleSystems = particleSystems.begin() ; it_particleSystems != particleSystems.end() ; ++it_particleSystems )
		delete (*it_particleSystems);
	particleSystems.clear();
	
	std::cout << "Scene deleted" << std::endl;
}

bool Scene :: create(const unsigned int limitNbParticles /* = 500000*/)
{
	unsigned int nbParticles = (limitNbParticles/30)+1;

	if ( !initGL() )
		return false;
	
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0.5f, 1, 0.5f), 15, false, 2));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-0.5f, 1, -0.5f), 15, false, 2));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-0.5f, 1, 0.5f), 15, false, 4));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0.5f, 1, -0.5f), 15, false, 4));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(1, 1, 0), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-1, 1, 0), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0, 1, -1), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0, 1, 1), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(2, 1, -2), 15, false, 0));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-2, 1, 2), 15, false, 0));
	
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-1.5f, 1, 1.5f), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-1.5f, 1, 0), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-1.5f, 1, -1.5f), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0, 1, -1.5f), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(1.5f, 1, -1.5f), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(1.5f, 1, 0), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(1.5f, 1, 1.5f), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0, 1, 1.5f), 15, false, 16));
	
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-2, 1, 3), 15, false, 2));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-3, 1, 2), 15, false, 2));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(2, 1, -3), 15, false, 2));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(3, 1, -2), 15, false, 2));
	
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-4, 1, -4), 15, true, 0));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(4, 1, 4), 15, true, 0));
	
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-2, 1, -4), 15, true, 4));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0, 1, -4), 15, false, 4));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(2, 1, -4), 15, true, 4));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(-2, 1, 4), 15, true, 4));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(0, 1, 4), 15, false, 4));
	particleSystems.push_back(new ParticleSystem(nbParticles, Math::Vertex3(2, 1, 4), 15, true, 4));
	
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, 5), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, -5), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, -5), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, 5), 15, false, 16));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(0, 1, 5), 15, true, 16));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, 0), 15, true, 16));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(0, 1, -5), 15, true, 16));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, 0), 15, true, 16));
	
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-3, 1, -5), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-2, 1, -5), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-1, 1, -5), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(1, 1, -5), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(2, 1, -5), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(3, 1, -5), 15, true, 8));
	
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-3, 1, 5), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-2, 1, 5), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-1, 1, 5), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(1, 1, 5), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(2, 1, 5), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(3, 1, 5), 15, true, 8));

	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, -3), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, -2), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, -1), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, 1), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, 2), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(-5, 1, 3), 15, true, 8));
	
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, -3), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, -2), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, -1), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, 1), 15, true, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, 2), 15, false, 8));
	particleSystems.push_back(new ParticleSystem(5000, Math::Vertex3(5, 1, 3), 15, true, 8));

	return true;
}

bool Scene :: initGL(void)
{
#ifdef _DEBUG
	glClearColor(0.2f, 0.0f, 0.0f, 0.3f);    // Set the background colour
#else
	glClearColor(0, 0, 0, 0);
#endif
    
    glClearDepth(1.0);	// Set the default value for the depth
	
    glDepthFunc(GL_LESS);	// Set the function for the depth
    glEnable(GL_DEPTH_TEST);	// Enable the depth test

	glPointSize(1.0f);	// Change the size of particles
    
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();          
    
    gluPerspective(45.0f,(float)glutGet(GLUT_WINDOW_WIDTH)/(float)glutGet(GLUT_WINDOW_HEIGHT),0.1f,80.0f);	// Set the perspective that OpenGL must apply
    
    glMatrixMode(GL_MODELVIEW);
	
	
	if ( vertexShader->load("Data/Shaders/simple.vert") )
	{
		if ( !vertexShader->compile() )
			return false;
		programShader->attach(vertexShader->getGLShaderId());
		if ( ! programShader->link() )
			return false;
	}
	else
		return false;
	
	if ( fSBlinking->load("Data/Shaders/blink.frag") )
	{
		if ( !fSBlinking->compile() )
			return false;
		programShader->attach(fSBlinking->getGLShaderId());
		if ( ! programShader->link() )
			return false;
	}
	else 
		return false;
	
	std::cout << "OpenGL Initialised" << std::endl;
	return true;
}

void Scene :: draw()
{	
	static int id_psId = programShader->getUniformId("psId");
	unsigned int nbParticles = 0;
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); // Clear the screen buffer and the depth buffer
	glLoadIdentity();
	
	cam->apply();		// Set the camera (viewed area by the window)
	
#ifdef _DEBUG			
	glBegin(GL_LINES);
		glColor3f(1, 0, 0);
		
		glVertex3f(0, 0, 0);
		glVertex3f(1, 0, 0);
		
		glColor3f(0, 1, 0);
		
		glVertex3f(0, 0, 0);
		glVertex3f(0, 1, 0);
		
		glColor3f(0, 0, 1);
		
		glVertex3f(0, 0, 0);
		glVertex3f(0, 0, 1);
    glEnd();
#endif
				
	programShader->use();
	
	for ( std::vector<ParticleSystem*>::iterator it_particleSystems = particleSystems.begin() ; it_particleSystems != particleSystems.end() ; ++it_particleSystems )
	{
		glUniform1i(id_psId, (*it_particleSystems)->getIndex());
		(*it_particleSystems)->draw();
		nbParticles += (*it_particleSystems)->getNbParticles();
	}
	
	programShader->unuse();
	
	Text::renderString(1,20,"Number of particles: " + Utils::toString(nbParticles), Colour3(1,0,0));
	
	glutSwapBuffers();
}

void Scene :: update(const unsigned int time) 
{
	static int id_blink = programShader->getUniformId("blinks");
	unsigned int nbParticles = 0;

	if ( !readyToQuit )		// While the FMOD play the music
	{
		(Music::getInstance()).update();
	
		(Music::getInstance()).getSpectrum(musicSpectrum, SPECTRUM_PRECISION); 
	}
	
	// Sets the shader variables
	programShader->use();

	glUniform1fv(id_blink, SPECTRUM_PRECISION*sizeof(float), musicSpectrum);	
	
	programShader->unuse();
	
	
	for ( std::vector<ParticleSystem*>::iterator it_particleSystems = particleSystems.begin() ; it_particleSystems != particleSystems.end() ; ++it_particleSystems )
	{
		// Update the birth rates of the particle systems, with the song spectrum
		(*it_particleSystems)->setBirthRate((unsigned int)(musicSpectrum[(*it_particleSystems)->getIndex()]*750));
		// Update the particle systems
		(*it_particleSystems)->update(time);
		nbParticles += (*it_particleSystems)->getNbParticles();
	}

	if ( readyToQuit && nbParticles == 0 )
		callback::cleanOut();
}
