#include "Texture.hpp"

#include "Visualisation.hpp"

#ifdef _SDL
	#include <SDL/SDL_image.h>
#endif

#include <iostream>

#ifdef _SDL

	/**
		Function for load a picture to create a texture
		@params const string name : the path of the picture
	*/
	Texture :: Texture ( const string& name )
	{
		SDL_Surface* temp = NULL;
		if ( (temp = IMG_Load(name.c_str())) == NULL )
		{
			std::cerr << "Output during loading the picture file [ FILE :" << name << std::endl;
			data=NULL;
		}
		else
		{		
			data = SDL_DisplayFormat(temp);
				
			width = data->w;
			height = data->h;
		}
		
		SDL_FreeSurface(temp);
	}

	/**
		Function to load a picture to create a texture , you can define a alpha colour
		@params const string name : the path of the picture
		@params const HAPI_TColour& transparancy : the alpha colour
	*/
	Texture :: Texture ( const string& name, const blackbox::TColour& colour)
	{
		SDL_Surface* temp = NULL;
		if ( (temp = IMG_Load(name.c_str())) == NULL )
		{
			std::cerr << "Output during loading the picture file [ FILE :" << name << std::endl;
			data=NULL;
		}
		else
		{		
			data = SDL_DisplayFormat(temp);
				
			width = data->w;
			height = data->h;
				
			if ( !setAlphaColour(colour) )
				std::cerr << "Output during the setting of the alpha colour" << std::endl;
		}
			
		SDL_FreeSurface(temp);
	}

	/**
		Function to transform the texture into a texture with transparancy using an alpha colour
		@params const HAPI_TColour& transparancy : the alpha colour
	*/
	bool Texture :: setAlphaColour(const blackbox::TColour& colour)
	{
		if ( data != NULL )
			return ((SDL_SetColorKey(data,SDL_SRCCOLORKEY,blackbox::getSDLColour(colour)))+1);
		return false;
	}
	
	/**
	Function to apply the texture like a background ( need to have the same dimension than the window )
	@return bool : if you can't apply this texture to the window
	*/
	bool Texture :: setBackground()
	{
		if ( data != NULL )
			if ( (unsigned)data->w == width && (unsigned)data->h == height )
			#ifdef _SDL
			{
				SDL_Rect pos = { 0,0 };
				SDL_BlitSurface(data, NULL, ((Visualisation::getVisualisation())->getSDL_window()), &pos);
				return true;
			}
			#endif
		return false;
	}

	/**
		Function to draw the Texture into the window
		@params const int x : the position in the x axis of the texture
		@params const int y : the position in the y axis of the texture
	*/
	bool Texture :: drawTexture(const int x/*=0*/, const int y/*=0*/)const
	{
		if( data != NULL )
		{
			SDL_Rect pos = { x,y };

			return ( SDL_BlitSurface(data, NULL, ((Visualisation::getVisualisation())->getSDL_window()), &pos)+1 );
		}
		return false;
	}

	/**
		Function to draw the Texture into the window
		@params const int x : the position in the x axis of the texture
		@params const int y : the position in the y axis of the texture
		@params const unsigned int source_start_x : position into the data of the texture where we must start to cpy in the x axis
		@params const unsigned int source_start_y : position into the data of the texture where we must start to cpy in the y axis
		@params const unsigned int width : size of the part of the data that we must copy
		@params const unsigned int height : size of the part of the data that we must copy
	*/
	bool Texture :: drawTexture(const int x, const int y,  const unsigned int source_start_x , const unsigned int source_start_y , const unsigned int width , const unsigned int height)const
	{
		SDL_Rect pos_dst = { source_start_x , source_start_y , width , height };
		SDL_Rect pos_src = { x , y };

		return ( SDL_BlitSurface(data, &pos_dst, ((Visualisation::getVisualisation())->getSDL_window()), &pos_src)+1);
	}

	/**
		Free the memory
	*/
	Texture :: ~Texture ()
	{
		if ( data != NULL )
			SDL_FreeSurface(data);
		data=NULL;
	}

#endif

#ifdef _HAPI

	/**
		Function for load a picture to create a texture
		@params const string name : the path of the picture
	*/
	Texture :: Texture (const string& name )
	{
		int width=0;
		int height=0;
		
		if ( !HAPI->LoadTexture(name, &data, &width, &height) )
		{
			HAPI->UserMessage("Unable to load the texture file","Output!",eButtonTypeOk );
			data=NULL;
		}
		
		this->width = width;
		this->height = height;

		alpha = false;
	}
	
	/**
		Function to load a picture to create a texture , you can define a alpha colour
		@params const string name : the path of the picture
		@params const HAPI_TColour& transparancy : the alpha colour
	*/
	Texture :: Texture (const string& name,const blackbox::TColour& transparancy)
	{
		int width=0;
		int height=0;
		
		if ( !HAPI->LoadTexture(name, &data, &width, &height) )
		{
			HAPI->UserMessage("Unable to load the texture file","Output!",eButtonTypeOk );
			data=NULL;
		}
		
		this->width = width;
		this->height = height;
			
		setAlphaColour(transparancy);
		alpha = true;
	}
	
		/**
		Function to create a Texture from a data
		@params const unsigned char *data : the data to create the texture
		@params const int width : the width of the texture
		@params const int height : the height of the texture
	*/
	Texture :: Texture (const unsigned char *data, const int width, const int height):width(width),height(height)
	{
		this->data = new unsigned char[width*height*4];
		
		memcpy(this->data,data,width*height*4);
	}

	/**
		Function to create a Texture from a data with an alpha colour
		@params const unsigned char *data : the data to create the texture
		@params const int width : the width of the texture
		@params const int height : the height of the texture
		@params const HAPI_TColour& transparancy : the alpha colour
	*/
	Texture :: Texture (const unsigned char *data, const int width, const int height,const blackbox::TColour& transparancy):width(width),height(height)
	{
		this->data = new unsigned char[width*height*4];
		
		memcpy(this->data,data,width*height*4);
	
		setAlphaColour(transparancy);
	}
	
	/**
		Function to transform the texture into a texture with transparancy using an alpha colour
		@params const HAPI_TColour& transparancy : the alpha colour
	*/
	bool Texture ::  setAlphaColour(const blackbox::TColour& transparancy)
	{
		for ( unsigned int i = 0 ; i < width*height*4 ; i+=4 )
		{
			if ( transparancy.blue == data[i+2] && transparancy.green == data[i+1] && transparancy.red == data[i] )
				data[i+3] = 0;
		}
		return true;
	}
	
	/**
		Function to apply the texture like a background ( need to have the same dimension than the window )
		@return bool : if you can't apply this texture to the window
	*/
	bool Texture :: setBackground()
	{
		if ( data != NULL )
			if ( (Visualisation::getVisualisation())->getWidth() == width && (Visualisation::getVisualisation())->getHeight() == height )
				return (Visualisation::getVisualisation())->cpyToScreen(data);
		return false;
	}
	
	/**
		Function to draw the Texture into the window
		@params const int x : the position in the x axis of the texture
		@params const int y : the position in the y axis of the texture
	*/
	bool Texture :: drawTexture (const int x , const int y) const
	{
		if ( data != NULL )
		{
			// Test for the clipping

			int new_x = x;
			int new_y = y;
			unsigned int new_pos_start_x = 0;
			unsigned int new_pos_start_y = 0;
			unsigned int new_width = width;
			unsigned int new_height = height;
			bool need_clip = false;

			if ( x < 0 )
			{
				new_x = 0;
				new_pos_start_x = abs(x);
				new_width = width + x;	// Better than -abs(x)
				need_clip=true;
			}
			if ( y < 0 )
			{
				new_y = 0;
				new_pos_start_y = abs(y);
				new_height = height + y;	// Better than -abs(y)
				need_clip=true;
			}
			if ( x+width >= (Visualisation::getVisualisation())->getWidth() )
			{
				new_width = (width - ( x+width - (Visualisation::getVisualisation())->getWidth() ))-1;
				need_clip=true;
			}
			if ( y+height >= (Visualisation::getVisualisation())->getHeight() )
			{
				new_height = (height - ( y+height - (Visualisation::getVisualisation())->getHeight() ))-1;
				need_clip=true;
			}
			if ( need_clip )
			{
				return drawTexture(new_x,new_y,new_pos_start_x,new_pos_start_y,new_width,new_height);
			}


			int dec_x = 0;
			int dec_y = y-1;

			for ( unsigned int i = 0 ; i < width * height * 4 ; i+=4 )
			{
				dec_x = ((i/4)%(width));
				if ( (i%(width*4)) == 0)
					dec_y++;
				(Visualisation::getVisualisation())->setPixel(x+dec_x, dec_y, blackbox::TColour(data[i+2], data[i+1] , data[i], data[i+3]) );
			}
			return true;
		}
		return false;
	}

	/**
		Function to draw the Texture into the window
		@params const int pos_x : the position in the x axis of the texture
		@params const int pos_y : the position in the y axis of the texture
		@params const unsigned int source_start_x : position into the data of the texture where we must start to cpy in the x axis
		@params const unsigned int source_start_y : position into the data of the texture where we must start to cpy in the y axis
		@params const unsigned int source_width : size of the part of the data that we must copy
		@params const unsigned int source_height : size of the part of the data that we must copy
	*/
	bool Texture :: drawTexture (const int pos_x , const int pos_y, const unsigned int source_start_x , const unsigned int source_start_y , const unsigned int source_width , const unsigned int source_height) const
	{
		if ( data != NULL )
		{
			if ( this->width - source_width < source_start_x || this->height - source_height < source_start_y || source_height > (Visualisation::getVisualisation())->getHeight() || source_width > (Visualisation::getVisualisation())->getWidth())
				return false;

			if ( !alpha )
			{
				for ( unsigned int i = 0 ; i < (unsigned)source_height ; ++i )
					 (Visualisation::getVisualisation())->cpyToPartOfScreen(data+((source_start_x+(height*source_start_y))*4)+(i*width*4) , pos_x , pos_y+i , source_width*4 );
				return true;
			}
			else
			{
				int dec_x = 0;
				int dec_y = pos_y-1;

				unsigned int offset = (source_start_x+(width*source_start_y))*4;
				unsigned int pitch = (width-source_width)*4;

				for ( unsigned int i = 0 ; i < source_width * source_height * 4 ; i+=4 )
				{
					dec_x = ((i/4)%(source_width));
					if ( (i%(source_width*4)) == 0)
						dec_y++;
					(Visualisation::getVisualisation())->setPixel(pos_x+dec_x, dec_y, blackbox::TColour(data[i+offset+((dec_y-pos_y)*(pitch))+2], data[i+offset+((dec_y-pos_y)*(pitch))+1], data[i+offset+((dec_y-pos_y)*(pitch))], data[i+offset+((dec_y-pos_y)*(pitch))+3]));
				}
				return true;
			}
		}
		return false;
	}
	
	/**
		Free the memory
	*/
	Texture :: ~Texture()
	{
		if ( data != NULL )
		{
			delete [] data;
			//delete data;
		}
	}
	
#endif
