#include "Level.hpp"

#include "World.hpp"
#include "World_Entities.hpp"
#include "Time_Manager.hpp"
#include "TextRender.hpp"
#include "Visualisation.hpp"
#include "Output.hpp"

#include <iostream>
#include <string>
#include <fstream>

#include <boost/filesystem.hpp>

extern bool somethingWrong;

/**
	Level's constructor
	Constructs the path for load the correct file.
	Loads the file
	@param unsigned char level : the number of the level to load
*/
Level :: Level(const unsigned char level):starting_time(blackbox::getTime()),offset(0)
{
	// Constructs the path
	boost::filesystem::path path("Data");
	path /= "Levels";

	// Constructs the file name
	std::string file_name = "Level_";
	std::string number = blackbox::toString((int)level);
	file_name = file_name + number + ".lev";

	// Append the folder and the file name
	boost::filesystem::path path_temp = path / file_name;

	if ( !loadFile(path_temp.string()) )
	{
		blackbox::showError("Output during loading the level [ LEVEL : " + blackbox::toString((int)level) + " ; Filename : " + path_temp.string());
		somethingWrong = true;
	}

	blackbox::showMessage("Level created");
}

/**
	File parser for load the level
	@param const std::string& name : the path of the file
	@param bool : if all it is right or not

	File format :

	path_for_the_first_background_texture
	path_for_the_second_background_texture

	time
	enemy_type
	x_position y_position

	.
	.
	.

	time
	enemy_type
	x_position y_position


	The time are in milliseconds
*/
bool Level :: loadFile(const std::string& path)
{
	// Open the file
	std::ifstream file(path.c_str());

	if ( !file.is_open() )	// The file is not correctly open
	{
		blackbox::showError("Output during opening the level file : [ FILE : " + path);
		return false;
	}

	/*
		Parsing
	*/

	// Reading the name of background files
	file >> text1;
	file >> text2;

	// Reading the enemies information
	while ( !file.eof() )	// Not the end
	{
		unsigned int time = 0;
		unsigned int type = 255;	// int to avoid a little bug ( conversion ascii -> int )
		int x = Visualisation::getVisualisation()->getWidth()+1, y = Visualisation::getVisualisation()->getHeight()+1;

		// Time
		file >> time;
		if ( time == 0 )
		{
			blackbox::showError("Output during reading the time ; file corrupt");
			return false;
		}

		// Type 
		file >> type;
		if ( type == 255 )
		{
			blackbox::showError("Output during reading the type ; file corrupt");
			return false;
		}

		// Position
		file >> x >> y;
		if ( x > (signed)Visualisation::getVisualisation()->getWidth() || y > (signed)Visualisation::getVisualisation()->getHeight())
		{
			blackbox::showError("Output during reading the position ; file corrupt");
			return false;
		}
		
		// Adding the element
		level_elements.push_back(new Level_Element(type+EntitiesType::SHIP1,Position(x,y),time));	// Type start from 0 in the file
	}

	file.close();

	return true; // It's ok
}

/**
	Level's destructor
*/
Level :: ~Level(void)
{
	for ( unsigned int i = 0 ; i < level_elements.size() ; ++i )
		delete level_elements[i];
	level_elements.clear();

	blackbox::showMessage("Level destructed");
}

/**
	Function to check if we have to create an enemy or not ; return if the level is ended
	@return bool : true if the level is finished
*/
bool Level :: update(void)
{
	if ( level_elements.size() == offset )
		return true;	// Level finished \o/

	for ( unsigned int i = offset ; i < level_elements.size() ; ++i )
	{
		if ( ( blackbox::getTime() - starting_time ) > level_elements[i]->time )
		{
			// We add the enemy
			World::getWorld()->addEnemy(level_elements[i]->type, level_elements[i]->pos);

			// We move the offset to not count this one after
			offset++;
		}
	}
	return false;
}
