#include "Animation.hpp"

#include "Time_Manager.hpp"

#include "global.hpp"

#ifdef _SDL
	#include <SDL/SDL.h>
	#include <SDL/SDL_image.h>
#endif
#ifdef _HAPI
	#include <HAPI_lib.h>
#endif

#include <iostream>
#include <string>

#ifdef _SDL

	/**
		Constructor able to create an Animation from a sprite composed with multiple texture
		@param const string& picture_name : name of the sprite
		@param const unsigned int size_sprite_x : size of one sprite on x
		@param const unsigned int size_sprite_y : size of one sprite on y
		@param const unsigned long tempo : tempo between two picture
	*/
	Animation :: Animation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y,const  unsigned long tempo):current_picture(0),tempo(tempo)
	{
		picture = new Texture(picture_name);
		
		if ( !picture->error() )
		{
			time = blackbox::getTime();

			size_x_sprite = size_sprite_x;
			size_y_sprite = size_sprite_y;
		
			offset_x = offset_y = 0;
			size_x = picture->getWidth();
			size_y = picture->getHeight();
			
			error = false;
		}
		else
		{
			delete picture;
			picture = NULL;
			error = true;
		}
	}

	/**
		Constructor able to create an Animation from a sprite composed with multiple texture
		@param const string& picture_name : name of the sprite
		@param const unsigned int size_sprite_x : size of one sprite on x
		@param const unsigned int size_sprite_y : size of one sprite on y
		@param const unsigned long tempo : tempo between two picture
		@param const blackbox::TColour& colour : colour for set the transparancy
	*/
	Animation :: Animation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y,const  unsigned long tempo,const blackbox::TColour& transparancy):current_picture(0),tempo(tempo)
	{
		picture = new Texture(picture_name,transparancy);
		
		if ( !picture->error() )
		{
			time = blackbox::getTime();

			size_x_sprite = size_sprite_x;
			size_y_sprite = size_sprite_y;
		
			offset_x = offset_y = 0;
			size_x = picture->getWidth();
			size_y = picture->getHeight();
			
			error = false;
		}
		else
		{
			delete picture;
			picture = NULL;
			error = true;
		}
	}

	/**
		Constructor able to create an Animation from a sprite composed with multiple texture
		@param const string& picture_name : name of the sprite
		@param const unsigned int size_sprite_x : size of one sprite on x
		@param const unsigned int size_sprite_y : size of one sprite on y
		@param const unsigned long tempo : tempo between two picture
		@param const unsigned int offset_x : offset on the x axis for start to pick the texture
		@param const unsigned int offset_y : offset on the y axis for start to pick the texture
		@param const unsigned int size_x : size on the x axis of the part of the texture selected
		@param const unsigned int size_y : size on the y axis of the part of the texture selected
	*/
	Animation :: Animation (const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y,const  unsigned long tempo,const unsigned int offset_x,const unsigned int offset_y,const unsigned int size_x,const unsigned int size_y):current_picture(0),tempo(tempo)
	{
		picture = new Texture(picture_name);
		
		if ( !picture->error() )
		{
			time = blackbox::getTime();

			size_x_sprite = size_sprite_x;
			size_y_sprite = size_sprite_y;
		
			this->offset_x = offset_x;
			this->offset_y = offset_y;
		
			this->size_x = size_x;
			this->size_y = size_y;
			
			error = false;
		}
		else
		{
			delete picture;
			picture = NULL;
			error = true;
		}
	}

	/**
		Constructor able to create an Animation from a sprite composed with multiple texture
		@param const string& picture_name : name of the sprite
		@param const unsigned int size_sprite_x : size of one sprite on x
		@param const unsigned int size_sprite_y : size of one sprite on y
		@param const unsigned long tempo : tempo between two picture
		@param const unsigned int offset_x : offset on the x axis for start to pick the texture
		@param const unsigned int offset_y : offset on the y axis for start to pick the texture
		@param const unsigned int size_x : size on the x axis of the part of the texture selected
		@param const unsigned int size_y : size on the y axis of the part of the texture selected
		@param const blackbox::TColour colour : the colour for set the transparancy
	*/
	Animation :: Animation (const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y,const  unsigned long tempo,const unsigned int offset_x,const unsigned int offset_y,const unsigned int size_x,const unsigned int size_y,const blackbox::TColour& transparancy):current_picture(0),tempo(tempo)
	{
		picture = new Texture(picture_name,transparancy);
		
		if ( !picture->error() )
		{
			time = blackbox::getTime();

			size_x_sprite = size_sprite_x;
			size_y_sprite = size_sprite_y;
		
			this->offset_x = offset_x;
			this->offset_y = offset_y;
		
			this->size_x = size_x;
			this->size_y = size_y;
			
			error = false;
		}
		else
		{
			delete picture;
			picture = NULL;
			error = true;
		}
	}

	/**
	draw the animation ( manage also the good running of the changement of the texture )
	@param const int x : position on x axis
	@param const int y : position on y axis
	*/
	void Animation :: drawAnimation(const int x, const int y)
	{
		unsigned int number_by_x = size_x/size_x_sprite;
		unsigned int number_by_y = size_y/size_y_sprite;
		
		unsigned int start_y = current_picture/number_by_x;
		
		picture->drawTexture(x,y,((current_picture%number_by_x)*size_x_sprite) + offset_x, (start_y*size_y_sprite)+offset_y, size_x_sprite, size_y_sprite);
		
		if ( tempo < (blackbox::getTime() - time) )
		{
			if ( current_picture < (number_by_x * number_by_y)-1 )
				current_picture++;
			else
				current_picture=0;
			time = blackbox::getTime();
		}	
	}

	/**
		draw the animation ( but just a part of the texture ) ( manage also the good running of the changement of the texture )
		@param const int x : position on x axis
		@param const int y : position on y axis
		@param const unsigned int source_start_x : the start point to use for draw the texture in x axis
		@param const unsigned int source_start_y : the start point to use for draw the texture in y axis
		@param const unsigned int width : the size of the part
		@param const unsigned int height : the size of the part
	*/
	void Animation :: drawAnimation(const int pos_x , const int pos_y, const unsigned int source_start_x , const unsigned int source_start_y , const unsigned int width , const unsigned int height)
	{
		unsigned int number_by_x = size_x/size_x_sprite;
		unsigned int number_by_y = size_y/size_y_sprite;
		
		unsigned int start_y = current_picture/number_by_x;
		
		picture->drawTexture(pos_x,pos_y,((current_picture%number_by_x)*size_x_sprite) + offset_x + source_start_x, (start_y*size_y_sprite)+offset_y + source_start_y, width, height);	

		if ( tempo < (blackbox::getTime() - time) )
		{
			if ( current_picture < (number_by_x * number_by_y)-1 )
				current_picture++;
			else
				current_picture=0;
			time = blackbox::getTime();
		}
	}

#endif

#ifdef _HAPI

	/**
	Constructor able to create an Animation from a sprite composed with multiple texture
	@param const string& picture_name : name of the sprite
	@param const unsigned int size_sprite_x : size of one sprite on x
	@param const unsigned int size_sprite_y : size of one sprite on y
	@param const unsigned long tempo : tempo between two picture
	*/
	Animation :: Animation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo):current_picture(0),tempo(tempo)
	{
		unsigned char* data=NULL;
		int width;
		int height;

		if ( !(HAPI->LoadTexture(picture_name,&data,&width,&height)) )
		{
			HAPI->UserMessage("The pictures not seems to exist","Fatal error!",eButtonTypeOk );
			delete [] data;
			error = true;
			return;
		}

		unsigned char* sprite_data = new unsigned char[size_sprite_y*size_sprite_x*4];

		// cutting of the texture for get all the sprite
		for ( unsigned int lines = 0 ; lines < height/size_sprite_y ; ++lines )
		{
			for ( unsigned int rows = 0 ; rows < width/size_sprite_x ; ++rows )
			{
				for ( unsigned int y = 0 ; y < size_sprite_y ; ++y )
					memcpy(sprite_data+( y * size_sprite_x*4) , data+( y * width*4)+ ( rows*size_sprite_x*4 ) + ( lines * size_sprite_y * width*4 ) , size_sprite_x*4);

				sequence.push_back(new Texture(sprite_data,size_sprite_x,size_sprite_y));
				
			}
		}	

		time = HAPI->GetTime();	

		delete [] sprite_data;
		
		error = false;
	}

	/**
		Constructor able to create an Animation from a sprite composed with multiple texture
		@param const string& picture_name : name of the sprite
		@param const unsigned int size_sprite_x : size of one sprite on x
		@param const unsigned int size_sprite_y : size of one sprite on y
		@param const unsigned long tempo : tempo between two picture
		@param const blackbox::TColour& colour : colour for set the transparancy
	*/
	Animation :: Animation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo,const blackbox::TColour& transparancy):current_picture(0),tempo(tempo)
	{
		unsigned char* data=NULL;
		int width;
		int height;

		if ( !(HAPI->LoadTexture(picture_name,&data,&width,&height)) )
		{
			HAPI->UserMessage("The pictures not seems to exist","Fatal error!",eButtonTypeOk );
			delete [] data;
			error = true;
			return;
		}

		unsigned char* sprite_data = new unsigned char[size_sprite_y*size_sprite_x*4];

		// cutting of the texture for get all the sprite
		for ( unsigned int lines = 0 ; lines < height/size_sprite_y ; ++lines )
		{
			for ( unsigned int rows = 0 ; rows < width/size_sprite_x ; ++rows )
			{
				for ( unsigned int y = 0 ; y < size_sprite_y ; ++y )
					memcpy(sprite_data+( y * size_sprite_x*4) , data+( y * width*4)+ ( rows*size_sprite_x*4 ) + ( lines * size_sprite_y * width*4 ) , size_sprite_x*4);

				sequence.push_back(new Texture(sprite_data,size_sprite_x,size_sprite_y,transparancy));
				
			}
		}	

		time = HAPI->GetTime();	

		delete [] sprite_data;
		delete [] data;
		
		error = false;
	}

	/**
		Constructor able to create an Animation from a sprite composed with multiple texture
		@param const string& picture_name : name of the sprite
		@param const unsigned int size_sprite_x : size of one sprite on x
		@param const unsigned int size_sprite_y : size of one sprite on y
		@param const unsigned long tempo : tempo between two picture
		@param const unsigned int offset_x : offset on the x axis for start to pick the texture
		@param const unsigned int offset_y : offset on the y axis for start to pick the texture
		@param const unsigned int size_x : size on the x axis of the part of the texture selected
		@param const unsigned int size_y : size on the y axis of the part of the texture selected
	*/
	Animation :: Animation (const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo,const unsigned int offset_x,const unsigned int offset_y,const unsigned int size_x,const unsigned int size_y):current_picture(0),tempo(tempo)
	{
		int width;
		int height;
		unsigned char* data=NULL;

		if ( !(HAPI->LoadTexture(picture_name,&data,&width,&height)) )
		{
			HAPI->UserMessage("The pictures not seems to exist","Fatal error!",eButtonTypeOk );
			delete [] data;
			error = true;
			return;
		}

		unsigned char* sprite_data = new unsigned char[size_sprite_y*size_sprite_x*4];

		// cutting of the texture for get all the sprite
		for ( unsigned int lines = 0 ; lines < size_y/size_sprite_y ; ++lines )
		{
			for ( unsigned int rows = 0 ; rows < size_x/size_sprite_x ; ++rows )
			{
				for ( unsigned int y = 0 ; y < size_sprite_y ; ++y )
					// We can easily optimise this loop by get out the multiplication, but it does not really matter because we are in a section run during the loading
					memcpy(sprite_data+( y * size_sprite_x*4) , data+( y * width*4)+ ( rows*size_sprite_x*4 ) + ( lines * size_sprite_y * width*4 ) + offset_x*4 + offset_y * width * 4 , size_sprite_x*4);

				sequence.push_back(new Texture(sprite_data,size_sprite_x,size_sprite_y));
				
			}
		}	

		time = HAPI->GetTime();	

		delete [] sprite_data;
		delete [] data;

		error = false;
	}

	/**
		Constructor able to create an Animation from a sprite composed with multiple texture
		@param const string& picture_name : name of the sprite
		@param const unsigned int size_sprite_x : size of one sprite on x
		@param const unsigned int size_sprite_y : size of one sprite on y
		@param const unsigned long tempo : tempo between two picture
		@param const unsigned int offset_x : offset on the x axis for start to pick the texture
		@param const unsigned int offset_y : offset on the y axis for start to pick the texture
		@param const unsigned int size_x : size on the x axis of the part of the texture selected
		@param const unsigned int size_y : size on the y axis of the part of the texture selected
		@param const blackbox::TColour colour : the colour for set the transparancy
	*/
	Animation :: Animation (const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo,const unsigned int offset_x,const unsigned int offset_y,const unsigned int size_x,const unsigned int size_y,const blackbox::TColour& transparancy):current_picture(0),tempo(tempo)
	{
		int width;
		int height;
		unsigned char* data=NULL;

		if ( !(HAPI->LoadTexture(picture_name,&data,&width,&height)) )
		{
			HAPI->UserMessage("The pictures not seems to exist","Fatal error!",eButtonTypeOk );
			delete [] data;
			error = true;
			return;
		}

		unsigned char* sprite_data = new unsigned char[size_sprite_y*size_sprite_x*4];

		// cutting of the texture for get all the sprite
		for ( unsigned int lines = 0 ; lines < size_y/size_sprite_y ; ++lines )
		{
			for ( unsigned int rows = 0 ; rows < size_x/size_sprite_x ; ++rows )
			{
				for ( unsigned int y = 0 ; y < size_sprite_y ; ++y )
					// We can easily optimise this loop by get out the multiplication, but it does not really matter because we are in a section run during the loading
					memcpy(sprite_data+( y * size_sprite_x*4) , data+( y * width*4)+ ( rows*size_sprite_x*4 ) + ( lines * size_sprite_y * width*4 ) + offset_x*4 + offset_y * width * 4 , size_sprite_x*4);

				sequence.push_back(new Texture(sprite_data,size_sprite_x,size_sprite_y,transparancy));
				
			}
		}	

		time = HAPI->GetTime();	

		delete [] sprite_data;
		delete [] data;

		error = false;
	}

	/**
	draw the animation ( manage also the good running of the changement of the texture )
	@param const int x : position on x axis
	@param const int y : position on y axis
	*/
	void Animation :: drawAnimation(const int x, const int y)
	{
		if ( sequence.size() > 0 )	// It exists
		{	
			if ( current_picture == sequence.size() )
				current_picture = 0;
			(sequence[current_picture])->drawTexture(x,y);

			if ( tempo < (blackbox::getTime() - time) )	// If the tempo is over, we change the texture
			{
				if ( current_picture < sequence.size() )
					current_picture++;
				else
					current_picture=0;
				time = (blackbox::getTime());
			}	
			
		}
	}

	/**
		draw the animation ( but just a part of the texture ) ( manage also the good running of the changement of the texture )
		@param const int x : position on x axis
		@param const int y : position on y axis
		@param const unsigned int source_start_x : the start point to use for draw the texture in x axis
		@param const unsigned int source_start_y : the start point to use for draw the texture in y axis
		@param const unsigned int width : the size of the part
		@param const unsigned int height : the size of the part
	*/
	void Animation :: drawAnimation(const int pos_x , const int pos_y, const unsigned int source_start_x , const unsigned int source_start_y , const unsigned int width , const unsigned int height)
	{
		if ( sequence.size() > 0 )	// It exists
		{	
			if ( current_picture == sequence.size() )	
				current_picture = 0;
			(sequence[current_picture])->drawTexture(pos_x,pos_y,source_start_x,source_start_y,width,height);

			if ( tempo < (blackbox::getTime() - time) )	// If the tempo is over, we change the texture
			{
				if ( current_picture < sequence.size() )
					current_picture++;
				else
					current_picture=0;
				time = (blackbox::getTime());
			}
		}
	}

#endif

/**
	Destructor
*/
#ifdef _SDL
	Animation :: ~Animation(void)
	{
		if ( picture != NULL )
			delete picture;
	}
#endif

#ifdef _HAPI
	Animation :: ~Animation(void)
	{
		for ( unsigned int i = 0 ; i < sequence.size() ; ++i )
			delete sequence[i];
	}
#endif
