#include "Visualisation.hpp"

#include "Keyboard.hpp"
#include "Time_Manager.hpp"

#include <iostream>
#include <string>

#include "exception.hpp"

#include "Texture.hpp"
#include "Animation.hpp"

#ifdef _SDL
	#include <SDL/SDL.h>
#endif

using namespace std;

Visualisation* Visualisation::window=NULL;

/**
	Get an instance of the class Visualisation ( singleton )
	@return Visualisation* : the instance
*/
Visualisation* Visualisation :: getVisualisation(void)
{
	if ( window == NULL )	// Not ready
	{
		try
		{
			window = new Visualisation(640,480,false);	// By default
		}
		catch (const string& error)
		{
			cerr << error << endl;
			return NULL;
		}
	}
	return window;
}

/**
	Get an instance of the class Visualisation ( singleton )
	@param const int width : the width of the new window
	@param const int height : the height of the new window
	@param const bool fullscreen : set the fullscreen mode
	@return Visualisation* : the instance
*/
Visualisation* Visualisation :: getVisualisation(const int width, const int height, const bool fullscreen)
{
	if ( window == NULL )
	{
		try
		{
			window = new Visualisation(width,height,fullscreen);
		}
		catch (const string& error)
		{
			cout << error << endl;
			return NULL;
		}
	}
	return window;
}

/**
	Destroy the instance ( singleton )
*/
void Visualisation :: destroyVisualisation(void)
{
	if ( window != NULL )
	{
		for ( int i = (int)listAnimation.size()-1 ; i >= 0 ; --i )
		{
			delete listAnimation[i];
			listAnimation.pop_back();
		}

		for ( int i = (int)listTexture.size()-1 ; i >= 0 ; --i )
		{
			delete listTexture[i];
			listTexture.pop_back();
		}
		delete window;
		window = NULL;
	}
}

#ifdef _SDL

	/**
		Constructor
		@param const int width : the width of the new window
		@param const int height : the height of the new window
		@param const bool fullscreen : set the fullscreen mode
	*/
	Visualisation :: Visualisation (const int width, const int height, const bool fullscreen)
	{
		if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
			throw InitException(1,"Unable to initialise the video",__FILE__,__LINE__);
			
		SDL_WM_SetCaption("BlacBox \\o/",NULL);

		SDL_EnableKeyRepeat(10, 10);
		//SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);

		if ( fullscreen )
			SDL_window = SDL_SetVideoMode(width,height,32,SDL_HWSURFACE|SDL_FULLSCREEN|SDL_DOUBLEBUF);
		else
			SDL_window = SDL_SetVideoMode(width,height,32,SDL_HWSURFACE|SDL_DOUBLEBUF);
			
		if ( SDL_window == NULL )
			throw InitException(1,"Unable to create the window",__FILE__,__LINE__);
			
		this->width = width;
		this->height = height;
		
		this->time_ellapsed = 0;
	}

	/**
		Function to display the scene , and check if the user click on the cross or press excape
		@param unsigned int refresh_time : time between two frames
		@return bool : return if we can continue the program or not
	*/
	bool Visualisation :: run(const unsigned int refresh_time/* = 0*/)
	{
		SDL_Flip(SDL_window);	// Display the screen

		if ( blackbox::getTime() - time_ellapsed < refresh_time )
			SDL_Delay(refresh_time - (blackbox::getTime() - time_ellapsed) );

		time_ellapsed = blackbox::getTime();
		return (blackbox::Keyboard::getKeyboard())->getEscape();
	}


	/**
		Function to clear the screen on black
	*/
	void Visualisation :: clearScreen(void)
	{
		SDL_FillRect(SDL_window,NULL,blackbox::getSDLColour(blackbox::TColour(0,0,0)));
	}

	/**
		Function to clear the screen with the colour
		@param const blackbox::TColour& colour : the new background colour
	*/
	void Visualisation :: clearScreen(const blackbox::TColour& colour)
	{
		SDL_FillRect(SDL_window,NULL,blackbox::getSDLColour(colour));
	}

	/**
		Function to get the colour of the pixel
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@return unsigned int : the colour
	*/
	unsigned int Visualisation :: getSDLPixel(const unsigned int x, const unsigned int y)
	{
		int bpp = SDL_window->format->BytesPerPixel;
		/* Here p is the address to the pixel we want to retrieve */
		Uint8 *p = (Uint8 *)SDL_window->pixels + y * SDL_window->pitch + x * bpp;

		switch(bpp) 
		{
			case 1:
				return *p;

			case 2:
				return *(Uint16 *)p;

			case 3:
				if(SDL_BYTEORDER == SDL_BIG_ENDIAN)
					return p[0] << 16 | p[1] << 8 | p[2];
				else
					return p[0] | p[1] << 8 | p[2] << 16;

			case 4:
				return *(Uint32 *)p;

			default:
				return 0;       /* shouldn't happen, but avoids warnings */
		}
	}

	/**
		Function to get the colour of the pixel
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@return blackbox::TColour : the colour
	*/
	blackbox::TColour Visualisation :: getPixel(const unsigned int x, const unsigned int y)
	{
		unsigned int colour = getSDLPixel(x,y);
		
		return blackbox::TColour(colour , colour << 8 , colour << 16);
	}

	/**
		Function to set a pixel with the colour
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@param const blackbox::TColour& colour : the colour
		@return bool : false -> something wrong happenned
	*/
	bool Visualisation :: setPixel(const unsigned int x , const unsigned int y , const blackbox::TColour& colour)
	{
		return setSDLPixel(x,y,blackbox::getSDLColour(colour));
	}

	/**
		Function to set a pixel with the colour
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@param const unsigned int colour : the colour
		@return bool : false -> something wrong happenned
	*/
	bool Visualisation :: setSDLPixel(const unsigned int x , const unsigned int y, const unsigned int colour)
	{
		 if ( SDL_MUSTLOCK(SDL_window) ) 
		 {
			if ( SDL_LockSurface(SDL_window) < 0 ) 
			{
				cerr << "Can't lock screen: " << SDL_GetError() << endl;
				return false;
			}
		}

		
		int bpp = SDL_window->format->BytesPerPixel;
		/* Here p is the address to the pixel we want to set */
		Uint8 *p = (Uint8 *)SDL_window->pixels + y * SDL_window->pitch + x * bpp;

		switch(bpp) {
		case 1:
			*p = colour;
			break;

		case 2:
			*(Uint16 *)p = colour;
			break;

		case 3:
			if(SDL_BYTEORDER == SDL_BIG_ENDIAN) {
				p[0] = (colour >> 16) & 0xff;
				p[1] = (colour >> 8) & 0xff;
				p[2] = colour & 0xff;
			} else {
				p[0] = colour & 0xff;
				p[1] = (colour >> 8) & 0xff;
				p[2] = (colour >> 16) & 0xff;
			}
			break;

		case 4:
			*(Uint32 *)p = colour;
			break;
		}
		
		if ( SDL_MUSTLOCK(SDL_window) )
		{
			SDL_UnlockSurface(SDL_window);
		}

		return true;
	}

	Visualisation :: ~Visualisation(void)
	{
		if ( SDL_window != NULL )
			SDL_FreeSurface(SDL_window);
		SDL_Quit();
		
		window = NULL;
	}

#endif

#ifdef _HAPI

	/**
		Constructor
		@param const int width : the width of the new window
		@param const int height : the height of the new window
		@param const bool fullscreen : set the fullscreen mode
	*/
	Visualisation :: Visualisation (const int width, const int height, const bool fullscreen)
	{
		int x=width;
		int y=height;
		
		if ( !HAPI->Initialise(&x , &y, fullscreen))
			throw InitException(1,"Unable to initialise the video",__FILE__,__LINE__);
			
		if ( (screen_data = HAPI->GetScreenPointer()) == NULL )
			throw InitException(1,"Unable to get the window pointer",__FILE__,__LINE__);
		else
			DW_screen_data = (DWORD*)screen_data;	
			
		this->width = x;
		this->height = y;
		
		this->time_ellapsed = 0;
	}

	/**
		Function to display the scene , and check if the user click on the cross or press escape
		This function can slow down the game ( number of FPS )
		@param unsigned int refresh_time : number of milliseconds to wait before the next frame ; by default we wait nothing
		@return bool : return if we can continue the program or not
	*/
	bool Visualisation :: run(const unsigned int refresh_time /*=0*/)
	{
		bool state = HAPI->Update();	// Display the screen

		while ( blackbox::getTime() - time_ellapsed < refresh_time ) {};

		time_ellapsed = blackbox::getTime();

		return (blackbox::Keyboard::getKeyboard())->getEscape() & state;
	}

	/**
		Function to clear the screen on black
	*/
	void Visualisation :: clearScreen(void)
	{
		memset(screen_data , 0 , (Visualisation::getVisualisation())->getWidth() * (Visualisation::getVisualisation())->getHeight() * 4 );
	}

	/**
		Function to clear the screen with the colour
		@param const blackbox::TColour& colour : the new background colour
	*/
	void Visualisation :: clearScreen(const blackbox::TColour& colour)
	{
		DWORD DWColour = (DWORD)(colour.red << 16 | colour.green << 8 | colour.blue );
		unsigned int size = sizeof(DWORD);

		unsigned screen_size = (Visualisation::getVisualisation())->getWidth() * (Visualisation::getVisualisation())->getHeight() * 4;
		unsigned size_last = 0;

		memcpy(DW_screen_data , &DWColour , size);

		while( size * 2 < screen_size )
		{
			memcpy ( screen_data + size , DW_screen_data , size);
			size *=2;
		}
		// To complete the filling
		memcpy ( screen_data + size , DW_screen_data , screen_size - size);
	}

	/**
		Function to get the colour of the pixel
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@return blackbox::TColour : the colour
	*/
	blackbox::TColour Visualisation :: getPixel(const unsigned int x, const unsigned int y)
	{
		unsigned int pixel = ( width * y + x)*4;
		return blackbox::TColour(screen_data[pixel+2],screen_data[pixel+1],screen_data[pixel]);
	}
	
	/**
		Function to set a pixel with the colour
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@param const unsigned char r : the red componant
		@param const unsigned char g : the green componant
		@param const unsigned char b : the blue componant
		@return bool : false -> something wrong happenned
	*/
	bool Visualisation :: setHAPIPixel(const unsigned int x, const unsigned int y, const unsigned char r, const unsigned char g, const unsigned char b)
	{
		if ( x >= width || y >= height )
		{
	#ifdef _DEBUG
			HAPI->DebugText("You try to put a pixel out the window !");
	#endif
			return false;
		}

		unsigned int pixel = (width * y + x)*4;

		screen_data[pixel] = b;
		screen_data[pixel+1] = g;
		screen_data[pixel+2] = r;

		return true;
	}
	
	/**
		Function to set a pixel with the colour
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@param const unsigned char r : the red componant
		@param const unsigned char g : the green componant
		@param const unsigned char b : the blue componant
		@param const unsigned char alpha : the alpha componant
		@return bool : false -> something wrong happenned
	*/
	bool Visualisation :: setHAPIPixel(const unsigned int x, const unsigned int y, const unsigned char r, const unsigned char g, const unsigned char b, const unsigned char alpha)
	{
		if ( x >= width || y >= height )
		{
		#ifdef _DEBUG
			HAPI->DebugText("You try to put a pixel out the window !");
		#endif
			return false;
		}
	
		unsigned int pixel = (width * y + x)*4;
	
		screen_data[pixel] += (alpha*(b-screen_data[pixel])>>8);
		screen_data[pixel+1] += ( alpha*(g-screen_data[pixel+1])>>8);
		screen_data[pixel+2] += ( alpha*(r-screen_data[pixel+2])>>8);
	
		return true;
	}

	/**
		Function to set a pixel with the colour
		@param const unsigned int x : position on the x axis
		@param const unsigned int y : position on the y axis
		@param const blackbox::TColour& colour : the colour
		@return bool : false -> something wrong happenned
	*/
	bool Visualisation :: setPixel(const unsigned int x , const unsigned int y , const blackbox::TColour& colour)
	{
		if ( colour.alpha == 255 )
			return setHAPIPixel(x,y,colour.red,colour.green,colour.blue);
		return setHAPIPixel(x,y,colour.red,colour.green,colour.blue,colour.alpha);
	}
	
	/**
		Function to copy data directly to the screen ( need to be the same size )
		@return bool : False if an error occurs
	*/
	bool Visualisation :: cpyToScreen(BYTE* const source)
	{
		if ( (memcpy(screen_data , source , width*height*4)) == NULL )
			return false;
		return true;
	}

	/**
		Function to copy data on a part of the screen
		@return bool : False if an error occurs
	*/
	bool Visualisation :: cpyToPartOfScreen(BYTE* const source, const unsigned int pos_x , const unsigned int pos_y , const unsigned int size)
	{
		if ( pos_x + size >= width && pos_y >= height )
			return false;
		if ( (memcpy(screen_data+((pos_x+(pos_y*width))*4), source, size)) == NULL )
			return false;
	return true;
	}

	Visualisation :: ~Visualisation(void)
	{
		if ( window != NULL )
			HAPI->Close();
		
		window = NULL;
	}

#endif

/**
	Function to add a new texture to the list
	@param const std::string& name : the path for get the texture
	@return int : the id of the texture
*/
int Visualisation :: addTexture(const std::string& name)
{
	listTexture.push_back(new Texture(name));
	
	return ((int)listTexture.size())-1;
}

/**
	Function to add a new texture to the list
	@param const std::string& name : the path for get the texture
	@param const blackbox::TColour& transparancy : the colour for set the transparancy
	@return int : the id of the texture
*/
int Visualisation :: addTexture(const std::string& name, const blackbox::TColour& transparancy)
{
	listTexture.push_back(new Texture(name,transparancy));
	
	return ((int)listTexture.size())-1;
}

/**
	draw a texture
	@param int id : to determine the texture
	@params const int x : the position in the x axis of the texture
	@params const int y : the position in the y axis of the texture
	@return bool : error
*/
bool Visualisation :: drawTexture(const int id, const int x, const int y) const
{
	if ( (unsigned)id < listTexture.size() && id != -1 )
	{
		listTexture[id]->drawTexture(x,y);
		return true;
	}
	return false;
}

/**
	draw a texture
	@param int id : to determine the texture
	@params const int pos_x : the position in the x axis of the texture
	@params const int pos_y : the position in the y axis of the texture
	@params const unsigned int source_start_x : position into the data of the texture where we must start to cpy in the x axis
	@params const unsigned int source_start_y : position into the data of the texture where we must start to cpy in the y axis
	@params const unsigned int source_width : size of the part of the data that we must copy
	@params const unsigned int source_height : size of the part of the data that we must copy
	@return bool : error
*/
bool Visualisation :: drawTexture(const int id, const int pos_x , const int pos_y, const unsigned int source_start_x , const unsigned int source_start_y , const unsigned int width ,  const unsigned int height)const
{
	if ( (unsigned)id < listTexture.size() && id != -1 )
	{
		listTexture[id]->drawTexture(pos_x,pos_y,source_start_x,source_start_y,width,height);
		return true;
	}
	return false;
}

/**
	Function for get the width of a texture
	@param const int id : the id of the texture
	@return unsigned int : width
*/
unsigned int Visualisation :: getTextureWidth(const int id)const
{
	if ( (unsigned)id < listTexture.size() && id != -1 )
		return listTexture[id]->getWidth();
	return 0;
}

/**
	Function for get the height of a texture
	@param const int id : the id of the texture
	@return unsigned int : height
*/
unsigned int Visualisation :: getTextureHeight(const int id)const
{
	if ( (unsigned)id < listTexture.size() && id != -1 )
		return listTexture[id]->getHeight();
	return 0;
}


/**
	add an animation
	@param const string& picture_name : name of the sprite
	@param const unsigned int size_sprite_x : size of one sprite on x
	@param const unsigned int size_sprite_y : size of one sprite on y
	@param const unsigned long tempo : tempo between two picture
	@return int id
*/
int Visualisation :: addAnimation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo)
{
	Animation* anim = new Animation(picture_name,size_sprite_x,size_sprite_y,tempo);
	if ( !anim->checkOutput() )
		listAnimation.push_back(anim);

	return ((int)listAnimation.size())-1;
}

/**
	add an animation
	@param const string& picture_name : name of the sprite
	@param const unsigned int size_sprite_x : size of one sprite on x
	@param const unsigned int size_sprite_y : size of one sprite on y
	@param const unsigned long tempo : tempo between two picture
	@param const blackbox::TColour& colour : colour for set the transparancy
	@return int id
*/
int Visualisation :: addAnimation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo,const unsigned int offset_x,const unsigned int offset_y,const unsigned int size_x,const unsigned int size_y)
{
	Animation* anim = new Animation(picture_name,size_sprite_x,size_sprite_y,tempo,offset_x,offset_y,size_x,size_y);
	if ( !anim->checkOutput() )
		listAnimation.push_back(anim);

	return ((int)listAnimation.size())-1;
}

/**
	add an animation
	@param const string& picture_name : name of the sprite
	@param const unsigned int size_sprite_x : size of one sprite on x
	@param const unsigned int size_sprite_y : size of one sprite on y
	@param const unsigned long tempo : tempo between two picture
	@param const unsigned int offset_x : offset on the x axis for start to pick the texture
	@param const unsigned int offset_y : offset on the y axis for start to pick the texture
	@param const unsigned int size_x : size on the x axis of the part of the texture selected
	@param const unsigned int size_y : size on the y axis of the part of the texture selected
	@return int id
*/
int Visualisation :: addAnimation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo, const blackbox::TColour& transparancy)
{
	Animation* anim = new Animation(picture_name, size_sprite_x,size_sprite_y,tempo,transparancy);
	if ( !anim->checkOutput() )
		listAnimation.push_back(anim);

	return ((int)listAnimation.size())-1;
}

/**
	add an animation
	@param const string& picture_name : name of the sprite
	@param const unsigned int size_sprite_x : size of one sprite on x
	@param const unsigned int size_sprite_y : size of one sprite on y
	@param const unsigned long tempo : tempo between two picture
	@param const unsigned int offset_x : offset on the x axis for start to pick the texture
	@param const unsigned int offset_y : offset on the y axis for start to pick the texture
	@param const unsigned int size_x : size on the x axis of the part of the texture selected
	@param const unsigned int size_y : size on the y axis of the part of the texture selected
	@param const blackbox::TColour colour : the colour for set the transparancy
	@return int id
*/
int Visualisation :: addAnimation(const string& picture_name, const unsigned int size_sprite_x, const unsigned int size_sprite_y, unsigned long tempo,const unsigned int offset_x,const unsigned int offset_y,const unsigned int size_x,const unsigned int size_y,const blackbox::TColour& transparancy)
{
	Animation* anim = new Animation(picture_name,size_sprite_x,size_sprite_y,tempo,offset_x,offset_y,size_x,size_y,transparancy);
	if ( !anim->checkOutput() )
		listAnimation.push_back(anim);

	return ((int)listAnimation.size())-1;
}

/**
	draw the animation
	@param const int x : position on x axis
	@param const int y : position on y axis
	@param unsigned int anim_number = NULL : for set the current picture of the animation
	@param unsinged int time = NULL : for set the current time of the animation
	@param unsigned int tempo = 0 : for change the tempo
	@return bool : error
*/
bool Visualisation :: drawAnimation(const int id, const int x/*=0*/ , const int y/*=0*/, unsigned int* anim_number/*=NULL*/, unsigned int* time/*=NULL*/, const unsigned int tempo/*=10000000*/  )const
{
	if ( (unsigned)id < listTexture.size() && id != -1 )
	{
		if ( anim_number != NULL )
			listAnimation[id]->setCurrent(*anim_number);
		if ( time != NULL )
			listAnimation[id]->setTime(*time);	
			
		listAnimation[id]->setTempo(tempo);

		listAnimation[id]->drawAnimation(x,y);
		
		if ( anim_number != NULL )	// Check if the texture change
			*anim_number = listAnimation[id]->getCurrent();
		if ( time != NULL )
			*time = listAnimation[id]->getTime();			
	
		return true;
	}
	return false;
}

/**
	draw the animation
	@param const int x : position on x axis
	@param const int y : position on y axis
	@param const unsigned int source_start_x : the start point to use for draw the texture in x axis
	@param const unsigned int source_start_y : the start point to use for draw the texture in y axis
	@param const unsigned int width : the size of the part
	@param const unsigned int height : the size of the part
	@param unsigned int anim_number = NULL : for set the current picture of the animation
	@param unsinged int time = NULL : for set the current time of the animation
	@param unsigned int tempo = 0 : for change the tempo
	@return bool : error
*/
bool Visualisation :: drawAnimation(const int id, const int pos_x , const int pos_y, const unsigned int source_start_x , const unsigned int source_start_y , const unsigned int width , const unsigned int height, unsigned int* anim_number/*=NULL*/, unsigned int* time/*=NULL*/, const unsigned int tempo/*=10000000*/  )const
{
	if ( (unsigned)id < listTexture.size() && id != -1 )
	{
		if ( anim_number != NULL )
			listAnimation[id]->setCurrent(*anim_number);
		if ( time != NULL )
			listAnimation[id]->setTime(*time);	

		listAnimation[id]->setTempo(tempo);

		listAnimation[id]->drawAnimation(pos_x,pos_y,source_start_x,source_start_y,width,height);

		if ( anim_number != NULL )	// Check if the texture change
			*anim_number = listAnimation[id]->getCurrent();
		if ( time != NULL )
			*time = listAnimation[id]->getTime();	
			
		return true;		
	}
	return false;
}
