#include "Keyboard.hpp"

#include <iostream>

blackbox::Keyboard* blackbox::Keyboard::keyboard=NULL;

/**
	Get an instance from the Keyboard ( singleton )
	@return blackbox::Keyboard* : the Keyboard's instance 
*/
blackbox::Keyboard* blackbox::Keyboard :: getKeyboard()
{
	if ( keyboard == NULL )
		keyboard = new blackbox::Keyboard();
	return keyboard;
}

/**
	Destroy the instance ( singleton )
*/
void blackbox::Keyboard :: destroyKeyboard()
{
	if ( keyboard != NULL )
	{
		delete keyboard;
		keyboard = NULL;
	}
}

#ifdef _SDL

	/**
		Constructor
	*/
	blackbox::Keyboard :: Keyboard()
	{
		event = new SDL_Event;	
	}

	/**
		Check if something to quit the program is pressed ( escape or cross )
		@return true if we must quit the program
	*/
	bool blackbox::Keyboard :: getEscape()
	{
		SDL_PollEvent(event);
		keyStates = SDL_GetKeyState( NULL );
		
		return !(event->type == SDL_QUIT || getKeyDown(BLACKBOX_ESCAPE));
	}

	/**
		Check if a key is pressed
		@param const ins key : the key to check
		@return bool if the key is pressed
	*/
	bool blackbox::Keyboard :: getKeyDown(const int key)
	{
		return keyStates[key];
	}

	/**
		Check if the Ctrl key is pressed
		@return bool if the key is pressed
	*/
	bool blackbox::Keyboard :: getModCtrlDown(void)
	{
		return SDL_GetModState() == KMOD_LCTRL || SDL_GetModState() == KMOD_RCTRL;
	}

	/**
		Check if the Shift key is pressed
		@return bool if the key is pressed
	*/
	bool blackbox::Keyboard :: getModShiftDown(void)
	{
		return SDL_GetModState() == KMOD_LSHIFT || SDL_GetModState() == KMOD_RSHIFT;
	}

	/**
		Check if the left click is pressed
		@return if the left click is pressed
	*/
	bool blackbox::Keyboard :: getLeftClick(void)
	{
		SDL_PollEvent(event);
		
		if ( event->button.type == SDL_MOUSEBUTTONDOWN )
			return event->button.button == SDL_BUTTON_LEFT;
		return false;
	}

	/**
		Check if the right click is pressed
		@return if the right click is pressed
	*/
	bool blackbox::Keyboard :: getRightClick(void)
	{
		SDL_PollEvent(event);
		
		if ( event->button.type == SDL_MOUSEBUTTONDOWN )
			return event->button.button == SDL_BUTTON_RIGHT;
		return false;
	}

	/**
		Function to get the position of the mouse
		@param int* x : the position on x axis
		@param int* y : the position on y axis
	*/
	void blackbox::Keyboard :: getMousePosition(int* x, int* y)
	{
		*x = event->button.x;
		*y = event->button.y;
	}

	/**
		Destructor
	*/
	blackbox::Keyboard :: ~Keyboard()
	{
		delete event;
		keyboard = NULL;
	}
#endif

#ifdef _HAPI

	/**
		Constructor
	*/
	blackbox::Keyboard :: Keyboard()
	{
		k_data = new HAPI_TKeyboardData();
		m_data = new HAPI_TMouseData();
	}

	/**
		Check if something to quit the program is pressed ( escape or cross )
		@return true if we must quit the program
	*/
	bool blackbox::Keyboard :: getEscape()
	{
		HAPI->GetKeyboardData(k_data);
		
		return !k_data->scanCode[HK_ESCAPE];
	}
	
	/**
		Check if a key is pressed
		@param const ins key : the key to check
		@return bool if the key is pressed
	*/
	bool blackbox::Keyboard :: getKeyDown(const int key)
	{
		HAPI->GetKeyboardData(k_data);
		
		return k_data->scanCode[key];
	}

	/**
		Check if the Ctrl key is pressed
		@return bool if the key is pressed
	*/
	bool blackbox::Keyboard :: getModCtrlDown(void)
	{
		return k_data->scanCode[HK_CONTROL];
	}

	/**
		Check if the Shift key is pressed
		@return bool if the key is pressed
	*/
	bool blackbox::Keyboard :: getModShiftDown(void)
	{
		return k_data->scanCode[HK_SHIFT];
	}

	/**
		Check if the left click is pressed
		@return if the left click is pressed
	*/
	bool blackbox::Keyboard :: getLeftClick(void)
	{
		HAPI->GetMouseData(m_data);
		
		return m_data->leftButtonDown;
	}

	/**
		Check if the right click is pressed
		@return if the right click is pressed
	*/
	bool blackbox::Keyboard :: getRightClick(void)
	{
		HAPI->GetMouseData(m_data);
		
		return m_data->rightButtonDown;
	}

	/**
		Function to get the position of the mouse
		@param int* x : the position on x axis
		@param int* y : the position on y axis
	*/
	void blackbox::Keyboard :: getMousePosition(int* x, int* y)
	{
		*x = m_data->x;
		*y = m_data->y;
	}

	/**
		Destructor
	*/
	blackbox::Keyboard :: ~Keyboard()
	{
		delete k_data;
		delete m_data;
		keyboard = NULL;
	}

#endif
