#include "Window.hpp"
#include "../Sound/Music.hpp"

#include <QApplication>
#include <QWidget>
#include <QMenuBar>
#include <QMenu>
#include <QGroupBox>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QMessageBox>
#include <QDir>
#include <QFileDialog>
#include <QStatusBar>
#include <QToolBar>
#include <QLabel>
#include <QSlider>
#include <QSpinBox>
#include <QTimer>
#include <QGridLayout>
#include <QScrollArea>

#include <iostream>

#define MAX_SPECTRUM_ANALYSED	64

const unsigned char widgetPlaces[MAX_SPECTRUM_ANALYSED][2] =
{
	{ 0, 0 },
	{ 0, 1 },
	{ 0, 2 },
	{ 0, 3 },
	{ 0, 4 },
	{ 0, 5 },
	{ 0, 6 },
	{ 0, 7 },
	{ 0, 8 },
	{ 0, 9 },
	{ 0, 10 },
	{ 0, 11 },
	{ 0, 12 },
	{ 0, 13 },
	{ 0, 14 },
	{ 0, 15 },
	
	{ 1, 0 },
	{ 1, 1 },
	{ 1, 2 },
	{ 1, 3 },
	{ 1, 4 },
	{ 1, 5 },
	{ 1, 6 },
	{ 1, 7 },
	{ 1, 8 },
	{ 1, 9 },
	{ 1, 10 },
	{ 1, 11 },
	{ 1, 12 },
	{ 1, 13 },
	{ 1, 14 },
	{ 1, 15 },
	
	{ 2, 0 },
	{ 2, 1 },
	{ 2, 2 },
	{ 2, 3 },
	{ 2, 4 },
	{ 2, 5 },
	{ 2, 6 },
	{ 2, 7 },
	{ 2, 8 },
	{ 2, 9 },
	{ 2, 10 },
	{ 2, 11 },
	{ 2, 12 },
	{ 2, 13 },
	{ 2, 14 },
	{ 2, 15 },
	
	{ 3, 0 },
	{ 3, 1 },
	{ 3, 2 },
	{ 3, 3 },
	{ 3, 4 },
	{ 3, 5 },
	{ 3, 6 },
	{ 3, 7 },
	{ 3, 8 },
	{ 3, 9 },
	{ 3, 10 },
	{ 3, 11 },
	{ 3, 12 },
	{ 3, 13 },
	{ 3, 14 },
	{ 3, 15 },
};

Window :: Window(QWidget* parent, const QString& title)
	:QMainWindow(parent)
{
	// Set the title	
	this->setWindowTitle(title);
	
	// Save the current path
	currentPath = QDir::homePath();
	
	// Create the file menu
	QMenu* fileMenu = new QMenu(tr("&File"), this);
	{	// File menu block
		this->addAction(fileMenu ,tr("&Open Song"), tr("Ctrl+O"), tr("Open a song file to analyse"), this, SLOT(openSong()));
	
		// Create a separator for the design
		fileMenu->addSeparator();
	
		this->addAction(fileMenu, tr("&Quit"), tr("Ctrl+Q"), tr("Quit the application"), qApp, SLOT(quit()));
	}
	
	// Create the music menu
	QMenu* musicMenu = new QMenu(tr("&Music"), this);
	{ 	// Music menu block
		playAction = this->addAction(musicMenu, tr("&Play"), tr("Ctrl+P"), tr("Play the song loaded"), this, SLOT(play()), false);
		pauseAction = this->addAction(musicMenu, tr("&Pause"), tr("Ctrl+P"), tr("Pause the song"), this, SLOT(pause()), false);
		stopAction = this->addAction(musicMenu, tr("&Stop"), tr("Ctrl+S"), tr("Stop the song"), this, SLOT(stop()), false);	
	}
	
	// Add the menu to the menu bar (window)
	this->menuBar()->addMenu(fileMenu);
	this->menuBar()->addMenu(musicMenu);
	// Create the about menu
	QAction* aboutAction = this->menuBar()->addAction("&?");
	connect(aboutAction, SIGNAL(triggered()), this, SLOT(aboutBox()));
	
	// Timer to update the song position (each second)
	updatePositionTimer = new QTimer(this);
	updatePositionTimer->setInterval(1000);
	
	connect(updatePositionTimer, SIGNAL(timeout()), this, SLOT(updateSongPosition()));
	
	// Main group box
	QFrame* centralBox = new QFrame(this);
	this->setCentralWidget(centralBox);
	
	QVBoxLayout* centralVBox = new QVBoxLayout(centralBox);
	
	// Song state
	QGroupBox* songStateBox = new QGroupBox(tr("Song state"),centralBox);
	centralVBox->addWidget(songStateBox);
	QVBoxLayout* songInfoBox = new QVBoxLayout(songStateBox);
	
	positionLabel = new QLabel("",this);
	positionLabel->setAlignment(Qt::AlignHCenter);
	songInfoBox->addWidget(positionLabel);
	
	positionSlider = new QSlider(Qt::Horizontal, this);
	positionSlider->hide();
	songInfoBox->addWidget(positionSlider);
	
	connect(positionSlider, SIGNAL(sliderMoved(int)), this, SLOT(seek(int)));
	
	// Analizer settings
	QGroupBox* analizerSettingsBox = new QGroupBox(tr("Analizer settings"), centralBox);
	centralVBox->addWidget(analizerSettingsBox);
	QHBoxLayout* analizerSettingsHBox = new QHBoxLayout(analizerSettingsBox);
	
	QVBoxLayout* nbFrequenciesBox = new QVBoxLayout(0);
	analizerSettingsHBox->addLayout(nbFrequenciesBox);
	
	QLabel* nbFrequenciesLabel = new QLabel(tr("Number frequencies analysed:"),this);
	nbFrequenciesBox->addWidget(nbFrequenciesLabel);
	
	nbFrequenciesSelector = new QSpinBox(this);
	nbFrequenciesSelector->setRange(0,MAX_SPECTRUM_ANALYSED);
	
	nbFrequenciesBox->addWidget(nbFrequenciesSelector);
	
	analizerSettingsHBox->addSpacing(10);
	
	QVBoxLayout* nbFrequencyMaxBox = new QVBoxLayout(0);
	analizerSettingsHBox->addLayout(nbFrequencyMaxBox);
	
	QLabel* nbFrequencyLabel = new QLabel(tr("Frequencies accuracy:"),this);
	nbFrequencyMaxBox->addWidget(nbFrequencyLabel);
	
	nbFrequencyMaxSelector = new QSpinBox(this);
	nbFrequencyMaxSelector->setRange(64,8192);
	nbFrequencyMaxSelector->setValue(64);
	nbFrequencyMaxBox->addWidget(nbFrequencyMaxSelector);
	
	// Analizer
	QGroupBox* analizerBox = new QGroupBox(tr("Analizer"), centralBox);
	analizerBox->setMaximumSize(1000, 600);
	centralVBox->addWidget(analizerBox);
	frequenciesBox = new QGridLayout(analizerBox);
	frequenciesBoxes = new QList<QVBoxLayout*>();
	frequenciesSliders = new QList<QSlider*>();
	frequenciesSpinBoxes = new QList<QSpinBox*>();
	
	nbFrequenciesSet = 0;
	
	updateAnalizerTimer = new QTimer(this);
	updateAnalizerTimer->setInterval(100);
	
	connect(updateAnalizerTimer, SIGNAL(timeout()), this, SLOT(updateAnalizer()));
	
	spectrum = new float[nbFrequencyMaxSelector->value()];
	
	connect(nbFrequencyMaxSelector, SIGNAL(valueChanged(int)), this, SLOT(updateFrequencyMaxSelector(int)));
	connect(nbFrequenciesSelector, SIGNAL(valueChanged(int)), this, SLOT(updateFrequenciesSelector(int)));
	
	nbFrequenciesSelector->setValue(4);	// This is done after the connect to directly create the objects
	
	// Update the status bar
	// statusBar()->setSizeGripEnabled(false);
	statusMessage = new QLabel(tr("Ready"));
	statusBar()->addWidget(statusMessage);
}

Window :: ~Window(void)
{	
	Music::getInstance().deleteInstance();
	
	delete playAction;
	delete pauseAction;
	delete stopAction;
	
	delete statusMessage;
	
	delete positionLabel;
	delete positionSlider;
	
	delete updatePositionTimer;
	
	delete nbFrequenciesSelector;
	delete nbFrequencyMaxSelector;
	
	for( QList<QVBoxLayout*>::iterator it_frequencies = frequenciesBoxes->begin() ; it_frequencies != frequenciesBoxes->end() ; ++it_frequencies )
		delete *it_frequencies;
	for( QList<QSlider*>::iterator it_frequencies = frequenciesSliders->begin() ; it_frequencies != frequenciesSliders->end() ; ++it_frequencies )
		delete *it_frequencies;
	for( QList<QSpinBox*>::iterator it_frequencies = frequenciesSpinBoxes->begin() ; it_frequencies != frequenciesSpinBoxes->end() ; ++it_frequencies )
		delete *it_frequencies;
	
	delete frequenciesBoxes;
	delete frequenciesSliders;
	delete frequenciesSpinBoxes;
	
	delete [] spectrum;
}

QAction* Window :: addAction(QMenu* parentMenu,const QString& entryName, const QString& keySequence, const QString& statusTipMessage, const QObject* object, const char* slot, const bool enabled)
{
	QAction* action = new QAction(0);
	action = parentMenu->addAction(entryName);
	action->setShortcut(QKeySequence(keySequence));
	action->setStatusTip(statusTipMessage);
	action->setEnabled(enabled);
	connect(action, SIGNAL(triggered()), object, slot);
	return action;
}

void Window :: openSong(void)
{
	QString fileName = QFileDialog::getOpenFileName(this,
         tr("Choose a song"), currentPath, tr("Songs (*.mp3 *.ogg *.wav)"));
	
	if ( !fileName.isEmpty() )
	{
		if ( Music::getInstance().loadFile(fileName.toStdString()) )
		{
			// Stop the timers
			updatePositionTimer->stop();
			updateAnalizerTimer->stop();			
			
			// Get some information about the length of the new song
			songMsLength = Music::getInstance().getMsLength();
			songSLength = songMsLength / 1000;
			songMLength = songSLength / 60;
			
			// Move to 0 the slider and hide it
			positionSlider->setMaximum(songSLength);
			positionSlider->setValue(0);
			positionSlider->hide();
			positionLabel->setText("");
			
			songSLength = songSLength % 60;

			// Get the song name			
			char* songName = Music::getInstance().getName();
			currentSong = QString(songName);
			delete [] songName;			
		
			// Enabled/Disabled the menu
			playAction->setEnabled(true);
			pauseAction->setEnabled(false);
			stopAction->setEnabled(false);
			
			// Update the status message
			statusMessage->setText(QString(tr("file: %1 Loaded")).arg(fileName));
			
			// update the current path by removing the filename of the path
			currentPath = fileName.remove(QRegExp("/*$"));
		}
	}
}

void Window :: aboutBox(void)
{
	QMessageBox::information(this, tr("About"), tr("\tFMOD Spectrum Analizer\n\nCreated by Alexandre LAURENT - Copyright 2009\n\tthedograge@hotmail.fr"));
}

void Window :: play(void)
{
	Music::getInstance().play();
	
	// Enabled/Disabled the menu
	playAction->setEnabled(false);
	pauseAction->setEnabled(true);
	stopAction->setEnabled(true);
	
	statusMessage->setText(QString(tr("Playing: %1")).arg(currentSong));
	
	// Set the position text
	positionLabel->setText(QString("0:00 / %1:%2").arg(songMLength).arg(songSLength,2,10,QLatin1Char('0')));

	// Shot the slider	
	positionSlider->show();
	
	// Enable the update of the position displayer
	updatePositionTimer->start();
	updateAnalizerTimer->start();
}

void Window :: pause(void)
{
	Music::getInstance().pause();
	
	// Enabled/Disabled the menu
	playAction->setEnabled(true);
	pauseAction->setEnabled(false);
	stopAction->setEnabled(true);
	
	statusMessage->setText(tr("Paused ..."));
	
	updatePositionTimer->stop();
	updateAnalizerTimer->stop();
}

void Window :: stop(void)
{
	Music::getInstance().stop();
	
	// Enabled/Disabled the menu	
	playAction->setEnabled(true);
	pauseAction->setEnabled(false);
	stopAction->setEnabled(false);
	
	statusMessage->setText(tr("Stopped"));
	
	// Enable the update of the position displayer
	updatePositionTimer->stop();
	positionLabel->setText("");
	
	updateAnalizerTimer->stop();
	
	positionSlider->hide();
}

void Window :: seek(int newValue)
{
	// The newValue is in second
	Music::getInstance().setMsPosition(newValue*1000);
}

void Window :: updateSongPosition(void)
{
	unsigned int songMsPosition = Music::getInstance().getMsPosition();
	unsigned int songSPosition = songMsPosition / 1000;
	unsigned int songMPosition = songSPosition / 60;
	
	// Update the slider
	positionSlider->setValue(songSPosition);
	
	songSPosition = songSPosition % 60;
	
	// Update the text
	positionLabel->setText(QString("%1:%2 / %3:%4").arg(songMPosition).arg(songSPosition,2,10,QLatin1Char('0')).arg(songMLength).arg(songSLength,2,10,QLatin1Char('0')));
	
	// Restart the timer
	updatePositionTimer->start();
	
	if ( songMsPosition == songMsLength )	// If we are at the end of the song, we stop
		this->stop();
}

void Window :: updateAnalizer(void)
{
	Music::getInstance().update();
	Music::getInstance().getSpectrum(spectrum, nbFrequencyMaxSelector->value());
	
	for ( unsigned int i = 0 ; i < nbFrequenciesSet ; ++i )
	{
		(*frequenciesSliders)[i]->setValue(static_cast<int>(spectrum[ (*frequenciesSpinBoxes)[i]->value() ] * 100));	
	}
	
	updateAnalizerTimer->start();
}

void Window :: updateFrequencyMaxSelector(int nbFrequencies)
{
	delete [] spectrum;
	spectrum = new float[nbFrequencies];
	
	for ( QList<QSpinBox*>::iterator it_frequencies = frequenciesSpinBoxes->begin() ; it_frequencies != frequenciesSpinBoxes->end() ; ++it_frequencies )
		(*it_frequencies)->setMaximum(nbFrequencies-1);
	
	if ( nbFrequencies % 4 != 0 )
		if ( ((nbFrequencies + 1) % 4) == 0 )	// So the new value is under a 2 power number (as 63 for 64)
			nbFrequencyMaxSelector->setValue((nbFrequencies+1)/2);
		else if ( ((nbFrequencies - 1) % 4) == 0 )	// So the new value is upper a 2 power number (as 65 for 64)
			nbFrequencyMaxSelector->setValue((nbFrequencies-1)*2);
			
}

void Window :: updateFrequenciesSelector(int nbSliders)
{
	while ( (unsigned)nbSliders < nbFrequenciesSet )
	{
		delete frequenciesSliders->last();
		frequenciesSliders->removeLast();
		
		delete frequenciesSpinBoxes->last();
		frequenciesSpinBoxes->removeLast();
		
		frequenciesBox->removeItem(frequenciesBoxes->last());	
		delete frequenciesBoxes->last();
		frequenciesBoxes->removeLast();
		
		nbFrequenciesSet--;
	}
	
	while ( (unsigned)nbSliders > nbFrequenciesSet )
	{
		this->addSpectrum();
		
		nbFrequenciesSet++;
	}
}

void Window :: addSpectrum(void)
{
	frequenciesBoxes->append(new QVBoxLayout(0));
	frequenciesBoxes->last()->setAlignment(Qt::AlignCenter);
	
	
	frequenciesSliders->append(new QSlider(Qt::Vertical, this));
	frequenciesSliders->last()->setRange(0,100);
	frequenciesSliders->last()->setMinimumHeight(100);
	frequenciesSliders->last()->setTickInterval(20);
	frequenciesSliders->last()->setTickPosition(QSlider::TicksBothSides);
	frequenciesSliders->last()->setEnabled(false);
	
	frequenciesSpinBoxes->append(new QSpinBox(this));
	frequenciesSpinBoxes->last()->setMaximum(nbFrequencyMaxSelector->value()-1);
	frequenciesSpinBoxes->last()->setValue(frequenciesSpinBoxes->count()-1);
	
	frequenciesBoxes->last()->addWidget(frequenciesSpinBoxes->last(), 0, Qt::AlignCenter);
	frequenciesBoxes->last()->addWidget(frequenciesSliders->last(), 0, Qt::AlignHCenter);
	
	frequenciesBox->addLayout(frequenciesBoxes->last(), widgetPlaces[nbFrequenciesSet][0], widgetPlaces[nbFrequenciesSet][1]);
}
