#include "Music.hpp"

#ifdef _DEBUG
	#include <iostream>
	#include <fmod_errors.h>
#endif

Music :: Music(void)
{	
	system = NULL;
	sound = NULL;
	channel = NULL;
	
#ifdef _DEBUG
	std::cout << "Music created" << std::endl;
#endif
}

Music :: ~Music(void)
{
	FMOD_Sound_Release(sound);
	FMOD_System_Close(system);
	FMOD_System_Release(system);
	
#ifdef _DEBUG
	std::cout << "Music deleted" << std::endl;
#endif
}

bool Music :: create(void)
{
	FMOD_RESULT result;
	
	result = FMOD_System_Create(&system);
	if ( result != FMOD_OK )
	{
#ifdef _DEBUG
		std::cerr << FMOD_ErrorString(result) << std::endl;
#endif
		return false;
	}
		
	result = FMOD_System_Init(system, 10, FMOD_INIT_NORMAL, NULL);
	if ( result != FMOD_OK )
	{
#ifdef _DEBUG
		std::cerr << FMOD_ErrorString(result) << std::endl;
#endif
		return false;
	}
	
	return true;
}

bool Music :: loadFile(const std::string& filename)
{
	FMOD_RESULT result;
	
	FMOD_Sound_Release(sound);
	sound = NULL;
	
	result = FMOD_System_CreateStream(system, filename.c_str(), FMOD_SOFTWARE | FMOD_CREATESAMPLE | FMOD_UNIQUE | FMOD_LOOP_OFF | FMOD_2D | FMOD_ACCURATETIME, 0, &sound);
	if ( result != FMOD_OK )
	{
#ifdef _DEBUG
		std::cerr << FMOD_ErrorString(result) << std::endl;
#endif
		return false;
	}
	
	return true;
}

void Music :: play(void)
{
	FMOD_BOOL paused = false;
	
	FMOD_Channel_GetPaused(channel, &paused);
	
	if ( paused )
		FMOD_Channel_SetPaused(channel, false);	
	else
		FMOD_System_PlaySound(system, FMOD_CHANNEL_FREE, sound, 0, &channel);
}

void Music :: pause(void)
{
	FMOD_Channel_SetPaused(channel, true);
}

void Music :: stop(void)
{
	FMOD_Channel_Stop(channel);
}

void Music :: update(void)
{
	FMOD_System_Update(system);
}

void Music :: getSpectrum(float* spectrum, const unsigned int precision)
{
	FMOD_Channel_GetSpectrum(channel, spectrum, precision, 0, FMOD_DSP_FFT_WINDOW_TRIANGLE);
}

unsigned int Music :: getMsLength(void) const
{
	FMOD_RESULT result;
	unsigned int length = 0;
	
	result = FMOD_Sound_GetLength(sound, &length, FMOD_TIMEUNIT_MS);
	if ( result != FMOD_OK )
	{
#ifdef _DEBUG
		std::cerr << FMOD_ErrorString(result) << std::endl;
#endif
	}
	
	return length;
}

unsigned int Music :: getMsPosition(void) const
{
	FMOD_RESULT result;
	unsigned int position = 0;
	
	result = FMOD_Channel_GetPosition(channel, &position, FMOD_TIMEUNIT_MS);
	if ( result != FMOD_OK )
	{
#ifdef _DEBUG
		std::cerr << FMOD_ErrorString(result) << std::endl;
#endif
	}
	return position;
}

void Music :: setMsPosition(const unsigned int newPosition)
{
	FMOD_RESULT result;
	
	result = FMOD_Channel_SetPosition(channel, newPosition, FMOD_TIMEUNIT_MS);
	if ( result != FMOD_OK )
	{
#ifdef _DEBUG
		std::cerr << FMOD_ErrorString(result) << std::endl;
#endif
	}
}

char* Music :: getName(void) const
{
	FMOD_RESULT result;
	char* name = new char[50];
	
	result = FMOD_Sound_GetName(sound, name, 50);
	if ( result != FMOD_OK )
	{
#ifdef _DEBUG
		std::cerr << FMOD_ErrorString(result) << std::endl;
#endif
	}
	return name;
}

