#include "World.hpp"

#include "World_Entities.hpp"
#include "UI.hpp"
#include "Level.hpp"
#include "Background.hpp"

#include "Player.hpp"

// Enemies
#include "Ship1.hpp"
#include "Ship2.hpp"
#include "Ship3.hpp"

// Bullets
#include "Bullet1.hpp"
#include "Bullet2.hpp"
#include "Bullet3.hpp"

// Explosions
#include "Explosion.hpp"

#include "Position.hpp"

#include "Collision.hpp"

#include "Visualisation.hpp"
#include "Time_Manager.hpp"
#include "TextRender.hpp"
#include "Sound.hpp"
#include "Output.hpp"

#include "Highscore.hpp"

#include <iostream>

World* World::world = NULL;

/**
	Constructor for the World
	( From nothing, something appears )
	@param unsigned char difficulty
*/
World :: World(const unsigned char difficulty)
{
	last_time = 0;
	next_time = 0;

	// First level
	level_number = 1;

	score = 0;
	highscore = loadHighScore();
	this->difficulty = difficulty;

	win=false;
	lose=false;

	wait_time = 0;

	// Load the sprites ( following the order of the enum 'Entities_Type')

	// Player
	sprites_ids.push_back(Visualisation::getVisualisation()->addTexture("Data/Gfx/Player.png", blackbox::TColour(255, 0 ,255 )));

	// Enemies
	sprites_ids.push_back(Visualisation::getVisualisation()->addTexture("Data/Gfx/Ship1.png", blackbox::TColour(255, 0 ,255 )));
	sprites_ids.push_back(Visualisation::getVisualisation()->addTexture("Data/Gfx/Ship2.png", blackbox::TColour(255, 0 ,255 )));
	sprites_ids.push_back(Visualisation::getVisualisation()->addTexture("Data/Gfx/Ship3.png", blackbox::TColour(255, 0 ,255 )));

	// Bullets
	sprites_ids.push_back(Visualisation::getVisualisation()->addAnimation("Data/Gfx/Bullet1.png",32,32,50,blackbox::TColour(255,0,255)));
	sprites_ids.push_back(Visualisation::getVisualisation()->addAnimation("Data/Gfx/Bullet2.png",16,16,50,blackbox::TColour(255,0,255)));
	sprites_ids.push_back(Visualisation::getVisualisation()->addAnimation("Data/Gfx/Bullet3.png",32,32,50,blackbox::TColour(255,0,255)));

	// Explosions
	sprites_ids.push_back(Visualisation::getVisualisation()->addAnimation("Data/Gfx/Explosion1.png",16,16,50,blackbox::TColour(255, 0 ,255 )));
	sprites_ids.push_back(Visualisation::getVisualisation()->addAnimation("Data/Gfx/Explosion2.png",32,32,50,blackbox::TColour(255,0,255)));

	// Load the sounds ( following the order of the enum 'Entities_Type')
	int temp_sound_id = 0;

	// Player
	sounds_ids.push_back(-1);

	// Enemies
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Ship1.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Ship2.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Ship3.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);

	// Bullets
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Bullet1.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Bullet2.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Bullet3.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);

	// Explosions
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Explosion1.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);
	(blackbox::Sound::getSound())->loadSound("Data/Sfx/Explosion2.wav",&temp_sound_id);
	sounds_ids.push_back(temp_sound_id);

	// Background
	background = NULL;
	// UI
	ui = new UI();
	// Level
	level =NULL;

	blackbox::showMessage("The world is created");
}

/**
	Destructor for the World
	( The end ... )
*/
World :: ~World(void)
{
	// Save highscore
	saveHighScore(highscore);

	for ( unsigned int i = 0 ; i < objects.size() ; ++i )
	{
		delete objects[i];
	}
	this->emptyTrash();

	if ( background != NULL )
		delete background;
	if ( ui != NULL )
		delete ui;
	if ( level != NULL )
		delete level;

	blackbox::showMessage("World deleted");
}

/**
	Getter for the World ( singleton )
	@param unsigned char difficulty =1 : the level of difficulty
*/
World* World :: getWorld(const unsigned char difficulty /* = 1*/)
{
	if ( world == NULL )
		world = new World(difficulty);
	return world;
}

/**
	Destructor for the World ( singleton )
*/
void World :: destroyWorld(void)
{
	delete world;
	world = NULL;	// The delete function do not set the pointer to NULL
}

/**
	To load a level
	Set the background texture
*/
void World :: loadLevel(void)
{
	//if ( level_number == 1)	// Just for the first
		addObject(EntitiesType::PLAYER,Position(Visualisation::getVisualisation()->getWidth()-(SIZE_UI+SIZE_SPRITES),Visualisation::getVisualisation()->getHeight()/2-(SIZE_SPRITES/2)));

	if ( level != NULL )
	{
		delete level;
		level = NULL;
	}
	level = new Level(level_number);	// Load the level

	if ( background != NULL )
	{
		delete background;
		background = NULL;
	}
	// Load the background
	background = new Background(level->getText1(), level->getText2());

	wait_time = 0;	// To reinitialise the wait_time between two levels
	win = false;
}

/**
	Function to draw all elements in the world
	( Colours appear on the screen )
	@return bool if we must stopped the world
*/
bool World :: draw(void)
{
	// Draw the background
	background->draw();

	for ( unsigned int i = 0 ; i < objects.size() ; ++i )
		objects[i]->draw();
	ui->draw();
	if ( lose )
	{
		ui->drawLose();
		if ( blackbox::getTime() -wait_time > 3000 )	// To not go directly to the menu
			return false;
	}
	else if ( win )
	{
		ui->drawWin();
		if ( blackbox::getTime() -wait_time > 3000 )	// To not go directly to the menu
		{
			if ( level_number == LEVEL_NUMBER)
				return false;
			else
			{
				// Frees the memory
				for ( unsigned int i = 0 ; i < objects.size()  ; ++i)
					delete objects[i];
				objects.clear();
				this->emptyTrash();

				level_number++;

				this->loadLevel();
			}
		}
	}
	return true;
}

/**
	Function to update the elements in the world
	( The world has life )
*/
void World :: update(void)
{	
	background->update();	// Move the background
	// Move objects
	for ( unsigned int i = 0 ; i < objects.size() ; ++i )
		objects[i]->update();
	this->doCollision();

	// Winning condition
	if ( level->update() )	// If true the enemies generation is finished
		if ( this->getEnemiesNumber() == 0 )	// No existing enemies
		{
			win = true;
			if ( wait_time == 0 )	// To avoid to set two times
				wait_time = blackbox::getTime();
		}
	this->updateTime();
	// Losing condition
	if ( !getPlayerIsAlive() )
	{
		lose = true;
		if ( wait_time == 0 )	// To avoid to set two times
			wait_time = blackbox::getTime();
	}

	// Score management :
	if ( score > highscore )
		highscore = score;
}

/**
	Function to update the keys time
*/
void World :: updateTime(void)
{
	 last_time = next_time;
	 next_time = blackbox::getTime();
}

/**
	Function to get the id of a sprite
	@param const unsigned char id : number of the sprite
	@return int : the id of the sprite ( point of view from Visualisation )
*/
int World :: getIdSprite(const unsigned char id)const
{
	if ( id < sprites_ids.size() )
		return sprites_ids[id];
	return -1;
}

/**
	Function to get the id of a sound
	@param const unsigned char id : number of the sound
	@return int : the id of the sound ( point of view from Sound )
*/
int World :: getIdSound(const unsigned char id)const
{
	if ( id < sounds_ids.size() )
		return sounds_ids[id];
	return -1;
}

/**
	Function to know the number of enemies into the World
	@return unsigned int : the number of enemies
*/
unsigned int World :: getEnemiesNumber(void)const
{
	unsigned int enemies_counter = 0;

	for ( std::vector<World_Entities*>::const_iterator it = objects.begin() ; it < objects.end() ; ++it )
	{
		unsigned char type = (*it)->getType();

		if ( type  > EntitiesType::PLAYER && type < EntitiesType::BULLET1 )
			enemies_counter++;
	}

	return enemies_counter;
}

/**
	Function to know if the player is still alive ( hope so :p )
	@return bool : true if the player is alive
*/
bool World :: getPlayerIsAlive(void)const
{
	if ( objects.size() != 0 )
		return objects[0]->getType() == EntitiesType::PLAYER; 	// I can do this king of test because i am sure that the first object created is the player
	return false;
}

/**
	Function to know the remaining life of the player
*/
unsigned int World :: getPLayerLife(void)const
{ 
	if ( objects.size() != 0 )
		if ( objects[0]->getType() == EntitiesType::PLAYER ) 
			return objects[0]->getLife();
	return 0;
}

/**
	Function to add object into the World
	( Some life appears here and there )
	@param const unsigned char type : the kind of the new object
	@param const Position& initial_pos : the start position
*/
void World :: addObject(const unsigned char type, const Position& initial_pos)
{
	World_Entities* object = NULL;

	switch ( type )
	{
		case EntitiesType::PLAYER:
			object = new Player(initial_pos);
			break;
		case EntitiesType::SHIP1:
			object = new Ship1(initial_pos);
			break;
		case EntitiesType::SHIP2:
			object = new Ship2(initial_pos);
			break;
		case EntitiesType::SHIP3:
			object = new Ship3(initial_pos);
			break;
		default:
			blackbox::showMessage("You try to create something unclassified, maybe an alien O_o !");
	}

	if ( object != NULL )
		objects.push_back(object);
}

/** 
	Function to add an enemy into the world
	@param const unsigned char type : type of the enemy
	@param const Position& : position of the new enemy
*/
void World :: addEnemy(const unsigned char type, const Position& initial_pos)
{
	if ( type > EntitiesType::PLAYER && type < EntitiesType::BULLET1 )
		this->addObject(type, initial_pos);
	else
		blackbox::showError("The enemy has not a good identifier [ TYPE : "+ blackbox::toString((int)type));
}

/** 
	Function to add an explosion into the world
	@param const unsigned char type : type of the explosion
	@param const Position& : position of the new explosion
*/
void World :: addExplosion(const unsigned char type, const Position& position)
{
	objects.push_back(new Explosion(position,type));
}

/**
	Function to add a bullet
	@param const unsigned char type : type of the new bullet
	@param const Position& position : position of the new bullet
	@param const unsigned int dx : direction on the x axis of the new bullet
	@param const unsigned int dy : direction on the y axis of the new bullet
*/
void World :: addBullet(const unsigned char type, const Position& position, const int dx, const int dy)
{
	World_Entities* bullet = NULL;

	switch ( type )
	{
		case EntitiesType::BULLET1:
			bullet = new Bullet1(position,dx,dy);
			break;
		case EntitiesType::BULLET2:
			bullet = new Bullet2(position,dx,dy);
			break;
		case EntitiesType::BULLET3:
			bullet = new Bullet3(position,dx,dy);
			break;
		default:
			blackbox::showMessage("Unknown bullet !");
	}
	if ( bullet != NULL )
		objects.push_back(bullet);
}

/**
	Function to empty the trash
*/
void World :: emptyTrash(void)
{
	for ( unsigned int i = 0 ; i < (unsigned)trash.size() ; ++i )
	{
		delete trash[i];	// Free the memory
	}
	trash.clear();	// clear the vector
}

/**
	Function to move an element into the trash
	@param const World_Entities* : the element to erase
*/
void World :: deleteElement(const World_Entities* elem)
{
	std::vector<World_Entities*>::iterator i = objects.begin();

	// Search for the element
	for ( ; (*i) != elem ; ++i ) {};	

	trash.push_back((*i));
	objects.erase(i);
}

/**
	Function to do the collisions and modify the state of the objects following these
	( Complexity : O((n-1)) )
*/
void World :: doCollision(void)
{
	for ( int i = (int)objects.size()-1 ; i >= 0 ; --i )
	{
		for ( int j = i ; j >= 0 ; --j )	// j always < i
		{
			if ( (unsigned)i < objects.size() && (unsigned)j < objects.size() )
			if ( getCollision(objects[i], objects[j]) )
			{
				// Collision between Player and Enemies
				if (objects[j]->getType() == EntitiesType::PLAYER )
				{
					if ( objects[i]->getType() > EntitiesType::PLAYER && objects[i]->getType() < EntitiesType::BULLET1 ) 
					{
						objects[i]->setDamage(1000);
						objects[j]->setDamage(1000);
						return; // Player dead nothing else to do ; game ended
					}
					// Collision between Bullet and Player
					else if ( objects[i]->getType() >= EntitiesType::BULLET1 && objects[i]->getType() < EntitiesType::EXPLOSION1 )
					{
						unsigned int damage = ((Bullet*)objects[i])->getDamage();
						this->addExplosion(EntitiesType::EXPLOSION1,Position(objects[i]->getPosX(),objects[i]->getPosY()));
						this->deleteElement(objects[i]);
						objects[j]->setDamage(damage);
					}
				}
				// Collision between Bullet and enemies
				else if ( objects[i]->getType() >= EntitiesType::BULLET1 || objects[j]->getType() >= EntitiesType::BULLET1 )
					if ( objects[i]->getType() < EntitiesType::EXPLOSION1 || objects[j]->getType() < EntitiesType::EXPLOSION1 )
						if (  objects[i]->getType() > EntitiesType::PLAYER || objects[j]->getType() > EntitiesType::PLAYER )
							if ( objects[i]->getType() < EntitiesType::BULLET1 || objects[j]->getType() < EntitiesType::BULLET1 )
							{

								unsigned int damage = 0;

								if ( objects[i]->getType() >= EntitiesType::BULLET1 && objects[i]->getType() < EntitiesType::EXPLOSION1 )
								{
									damage = ((Bullet*)objects[i])->getDamage();
									unsigned char enemy_type = objects[j]->getType();	// Save type because it can be deleted.
									if ( objects[j]->setDamage(damage) )	// If the object is deleted
									{
										increaseScore(enemy_type);
										i--;	// Because the vector lost one element and so my variable is no more correct
									}

									this->addExplosion(EntitiesType::EXPLOSION1,Position(objects[i]->getPosX(),objects[i]->getPosY()));
									this->deleteElement(objects[i]);
								}
								else
								{
									damage = ((Bullet*)objects[j])->getDamage();
									unsigned char enemy_type = objects[i]->getType();	// Save type because it can be deleted.
									if ( objects[i]->setDamage(damage) )
										increaseScore(enemy_type);
									this->addExplosion(EntitiesType::EXPLOSION1,Position(objects[j]->getPosX(),objects[j]->getPosY()));
									this->deleteElement(objects[j]);
									
								}
							}
			}
		}
	}
}

/**
	To increase the score
	@param const unsigned char type = type of the object who give the bonus
*/
void World :: increaseScore(const unsigned char type)
{
	switch ( type )
	{
		case EntitiesType::SHIP1:
			score += 20;
			break;
		case EntitiesType::SHIP2:
			score += 30;
			break;
		case EntitiesType::SHIP3:
			score += 25;
			break;
		case EntitiesType::PLAYER:
		score -= 100;
		break;
	}
}
