#include "TextRender.hpp"
#include <iostream>

#include <boost/filesystem/path.hpp>

#include "Visualisation.hpp"

blackbox::TextRender* blackbox::TextRender::textRender = NULL;

/**
	Get an instance of the TextRender ( singleton )
	@return blackbox::TextRender* : the instance
*/
blackbox::TextRender* blackbox::TextRender :: getTextRender()
{
	if ( textRender == NULL )
		textRender = new TextRender();
	return textRender;
}

/**
	Destruct the instance ( singleton )
*/
void blackbox::TextRender :: destroyTextRender()
{
	if ( textRender != NULL )
	{
		delete textRender;
		textRender = NULL;
	}
}

#ifdef _HAPI

	#include <HAPI_lib.h>

	/**
		Constructor
	*/
	blackbox::TextRender :: TextRender()
	{
		// Only for HAPI 1.18 : 
		HAPI->ChangeFont("Data/Font.ttf",SIZE_FONT,800);
	}

	/**
		Destructor
	*/
	blackbox::TextRender :: ~TextRender()
	{

	}

	/**
		Function to display a text
		@param const int x : position on the x axis
		@param const int y : position on the y axis
		@param const std::string& message : message to display
		@param const blackbox::TColour& colour : colour of the message
	*/
	void blackbox::TextRender :: renderText(const int x, const int y, const std::string& message,const blackbox::TColour& colour)
	{
		HAPI->RenderText(x,y,blackbox::getHAPIColour(colour),message);
	}	

#endif

#ifdef _SDL

	/**
		Constructor
	*/
	blackbox::TextRender :: TextRender()
	{
		if ( TTF_Init() == -1 )
			std::cerr << "Output during the initialisation of the TTF" << std::endl;
			
		font = TTF_OpenFont("Data/Font.ttf",SIZE_FONT-4);
		if ( font == NULL )
			std::cerr << "Output during the initialisation of the font" << std::endl;
	}

	/**
		Destructor
	*/
	blackbox::TextRender :: ~TextRender()
	{
		if ( font != NULL )
			TTF_CloseFont(font);

		TTF_Quit();
	}

	/**
		Function to display a text
		@param const int x : position on the x axis
		@param const int y : position on the y axis
		@param const std::string& message : message to display
		@param const blackbox::TColour& colour : colour of the message
	*/
	void blackbox::TextRender :: renderText(const int x, const int y, const std::string& message,const blackbox::TColour& colour)
	{
		if ( font != NULL )
		{
			SDL_Color color = { colour.red , colour.green, colour.blue ,colour.alpha };
			SDL_Surface* text = TTF_RenderText_Solid(font,message.c_str(),color);

			SDL_Rect pos = { x , y };
			
			if ( text != NULL )
			{
				SDL_BlitSurface(text,NULL,(Visualisation::getVisualisation())->getSDL_window(),&pos);

				SDL_FreeSurface(text);
			}
		}
	}

#endif
