#include "Sound.hpp"

#include <iostream>

#ifdef _HAPI
	#include <HAPI_lib.h>
#endif

blackbox::Sound* blackbox::Sound::sound=NULL;

/**
	Get an instance of the Sound ( singleton )
	@return blackbox::Sound* : the Sound's instance 
*/
blackbox::Sound* blackbox::Sound :: getSound()
{
	if ( sound == NULL )
		sound = new Sound();
	return sound;
}

/**
	To destroy the instance ( singleton )
*/
void blackbox::Sound :: destroySound()
{
	if ( sound != NULL )
	{
		delete sound;
		sound = NULL;
	}
}

#ifdef _SDL
	/**
		Constructor
	*/
	blackbox::Sound :: Sound()
	{
		if ( Mix_OpenAudio(MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 2, 1024) == -1 )
			std::cerr << "Unable to start the Audio system ( SDL )" << std::endl;
	}

	/**
		Destructor
	*/
	blackbox::Sound :: ~Sound()
	{
		for ( unsigned int i = 0 ; i < bank_sound.size() ; ++i )
			Mix_FreeChunk(bank_sound[i]);
		for ( unsigned int i = 0 ; i < bank_music.size() ; ++i )
			Mix_FreeMusic(bank_music[i]);
		Mix_CloseAudio();
	}

	/**
		Function to load a sound
		@param const std::string& pathname : the path where it get the sound
		@param int* id : the id to use the sound
	*/
	void blackbox::Sound :: loadSound(const std::string& pathname, int* const id)
	{
		Mix_Chunk* song = NULL;
		if ( ( song = Mix_LoadWAV(pathname.c_str())) == NULL )
		{
			std::cerr << "Unable to load the song : " << pathname << std::endl;	
		}
		
		bank_sound.push_back(song);

		*id = (int)bank_sound.size()-1;
	}

	/**
		Function to load a music
		@param const std::string& pathname : the path where it get the sound
		@param int* id : the id to use the sound
	*/		
	void blackbox::Sound :: loadMusic(const std::string& pathname, int* const id)
	{
		bank_music.push_back(Mix_LoadMUS(pathname.c_str()));

		*id = (int)bank_music.size()-1;
	}

	/**
		Function to play a sound
		@param const int num : the id of the sound
		@param const bool loop : if the sound need to loop or not
		@param const unsigned int volume : the volume of the sound
	*/
	void blackbox::Sound :: playSound(const int num, const bool loop,const unsigned int volume)
	{
		if ( num != -1 )
		{
			int channel = -1;
			Mix_Volume(channel,(128/1000.0)*volume);
			//std::cout << "Volume ? : " << ((128/1000.0)*volume) << std::endl;
			if ( loop )
				channel = Mix_PlayChannel(-1,bank_sound[num],-1);
			else
				channel = Mix_PlayChannel(-1,bank_sound[num],0);
		
			if ( channel == -1 )
				std::cerr << " An error occur for play the song " << std::endl;
		}
	}

	/**
		Function to play a music
		@param const int num : the id of the music
	*/
	void blackbox::Sound :: playMusic(const int num)
	{
		if ( num != -1 )
			Mix_PlayMusic(bank_music[num],-1);
	}

	/**
		Function to stop a sound
		@param const int num : the id of the sound to stop
	*/
	void blackbox::Sound :: stopSound(const int num)
	{
		Mix_HaltChannel(num);
	}
#endif

#ifdef _HAPI

	/**
		Constructor
	*/
	blackbox::Sound :: Sound() {}

	/**
		Destructor
	*/
	blackbox::Sound :: ~Sound()
	{
		for ( unsigned int i = 0 ; i < bank_sound.size() ; ++i )
			HAPI->StopSound(bank_sound[i]);
	}

	/**
		Function to load a sound
		@param const std::string& pathname : the path where it get the sound
		@param int* id : the id to use the sound
	*/
	void blackbox::Sound :: loadSound(const std::string& pathname, int* const id)
	{
		int id_sound = 0;

		if ( HAPI->LoadSound(pathname,&id_sound) )
		{
			bank_sound.push_back(id_sound);
		}

		*id = (int)bank_sound.size()-1;
	}

	/**
		Function to load a music
		@param const std::string& pathname : the path where it get the sound
		@param int* id : the id to use the sound
	*/	
	void blackbox::Sound :: loadMusic(const std::string& pathname, int* const id)
	{
		music = pathname;
		//*id=0;	// Don't care , useless with HAPI
	}

	/**
		Function to play a sound
		@param const int num : the id of the sound
		@param const bool loop : if the sound need to loop or not
		@param const unsigned int volume : the volume of the sound
	*/
	void blackbox::Sound :: playSound(const int num, const bool loop,const unsigned int volume)
	{
		if ( num != -1 )
			HAPI->PlayASound(bank_sound[num], loop,volume);
	}

	/**
		Function to play a music
		@param const int num : the id of the music
	*/
	void blackbox::Sound :: playMusic(const int num)
	{
		HAPI->PlayStreamedMedia(music);
	}

	/**
		Function to stop a sound
		@param const int num : the id of the sound to stop
	*/
	void blackbox::Sound :: stopSound(const int num)
	{
		if ( num != -1 )
			HAPI->StopSound(bank_sound[num]);
	}

#endif
